package dev.fritz2.lenses

//import net.ltgt.gradle.incap.IncrementalAnnotationProcessor
//import net.ltgt.gradle.incap.IncrementalAnnotationProcessorType
import com.squareup.kotlinpoet.*
import com.squareup.kotlinpoet.ParameterizedTypeName.Companion.plusParameter
import com.squareup.kotlinpoet.metadata.ImmutableKmClass
import com.squareup.kotlinpoet.metadata.KotlinPoetMetadataPreview
import com.squareup.kotlinpoet.metadata.isPrimary
import com.squareup.kotlinpoet.metadata.specs.toTypeSpec
import com.squareup.kotlinpoet.metadata.toImmutableKmClass
import dev.fritz2.lenses.LensesAnnotationProcessor.Companion.FRITZ2_VISIBILITY_OPTION_NAME
import dev.fritz2.lenses.LensesAnnotationProcessor.Companion.KAPT_KOTLIN_GENERATED_OPTION_NAME
import java.io.File
import javax.annotation.processing.*
import javax.lang.model.SourceVersion
import javax.lang.model.element.ElementKind
import javax.lang.model.element.TypeElement
import javax.tools.Diagnostic.Kind.ERROR


@KotlinPoetMetadataPreview
@SupportedAnnotationTypes("dev.fritz2.lenses.Lenses")
@SupportedSourceVersion(SourceVersion.RELEASE_8)
@SupportedOptions(KAPT_KOTLIN_GENERATED_OPTION_NAME, FRITZ2_VISIBILITY_OPTION_NAME)
//@IncrementalAnnotationProcessor(IncrementalAnnotationProcessorType.AGGREGATING)
class LensesAnnotationProcessor : AbstractProcessor() {
    companion object {
        const val KAPT_KOTLIN_GENERATED_OPTION_NAME = "kapt.kotlin.generated"
        const val FRITZ2_VISIBILITY_OPTION_NAME = "fritz2.visibility"
    }

    override fun process(annotations: MutableSet<out TypeElement>?, roundEnv: RoundEnvironment?): Boolean {
        val visibility = processingEnv.options[FRITZ2_VISIBILITY_OPTION_NAME]
        val kaptKotlinGeneratedDir = processingEnv.options[KAPT_KOTLIN_GENERATED_OPTION_NAME] ?: run {
            processingEnv.messager.printMessage(ERROR, "Can't find the target directory for generated Kotlin files.")
            return false
        }

//        val classInspector = ElementsClassInspector.create(processingEnv.elementUtils, processingEnv.typeUtils)
        val annotatedElements = roundEnv?.getElementsAnnotatedWith(Lenses::class.java)
        if (annotatedElements == null || annotatedElements.isEmpty()) return false

        val classesWithoutPackage = annotatedElements
            .asSequence()
            .filter { it.kind == ElementKind.CLASS }
            .map { it as TypeElement }
            .map { it.toImmutableKmClass() }
            .filterNot { it.name.contains('/') }
            .toList()

        if (classesWithoutPackage.isEmpty()) {
            annotatedElements
                .filter { it.kind == ElementKind.CLASS }
                .map { it as TypeElement }
                .associateWith { it.toTypeSpec() to it.toImmutableKmClass() }
                .toList()
                .groupBy { (element, _) ->
                    processingEnv.elementUtils.getPackageOf(element).qualifiedName.toString()
                }
                .forEach { (pack, classes) ->
                    val fileSpecBuilder = FileSpec.builder(pack, "GeneratedLenses")
                        .addComment("GENERATED by fritz2")
                        .addComment(" - NEVER CHANGE CONTENT MANUALLY!")
                        .addImport("dev.fritz2.lenses", "buildLens")

                    val lensesObject = TypeSpec.objectBuilder("L").addModifiers(visibility.asVisibilityModifier())
                    classes.forEach { (element, classData) ->
                        lensesObject.addType(handleDataClass(element, classData.first, classData.second))
                    }
                    fileSpecBuilder.addType(lensesObject.build())

                    fileSpecBuilder
                        .build()
                        .writeTo(File(kaptKotlinGeneratedDir).apply {
                            mkdir()
                        })
                }
            return true
        } else {
            processingEnv.messager.printMessage(
                ERROR,
                "@Lenses annotation can't be applied to data classes outside of a package! " +
                        "Please add package declarations to the following classes: ${
                            classesWithoutPackage.joinToString(", ") { it.name }
                        }"
            )
            return false
        }
    }

    private fun String?.asVisibilityModifier(default: KModifier = KModifier.PUBLIC): KModifier = setOf(
        KModifier.PUBLIC,
        KModifier.PROTECTED,
        KModifier.PRIVATE,
        KModifier.INTERNAL,
    ).find { it.name.equals(this, true) } ?: default

    private fun handleDataClass(element: TypeElement, classData: TypeSpec, kmClassData: ImmutableKmClass): TypeSpec {
        val classSpec = TypeSpec.objectBuilder(classData.name ?: "unknown")
        val ctorParamNames = kmClassData.constructors.first { it.isPrimary }.valueParameters.map { it.name }
        classData.propertySpecs
            .filter { ctorParamNames.contains(it.name) }
            .forEach { propertyData ->
                //FIXME: replace deprecated function call
                classSpec.addProperty((handleField(element.asClassName(), propertyData)))
            }

        return classSpec.build()
    }

    private fun handleField(className: ClassName, propertyData: PropertySpec): PropertySpec {

        val attributeName = propertyData.name

        val lensTypeName = ClassName("dev.fritz2.lenses", "Lens")
            //.plusParameter(ClassName("kotlin","String"))
            .plusParameter(className)
            .plusParameter(propertyData.type)

        return PropertySpec.builder(attributeName, lensTypeName)
            .initializer("buildLens(%S, { it.$attributeName }, { p, v -> p.copy($attributeName = v)})", attributeName)
            .build()
    }

}
