package dev.fritz2.headless.validation

import dev.fritz2.core.Inspector
import dev.fritz2.validation.Validation
import dev.fritz2.validation.ValidationMessage

/**
 * Enum which specify the [Severity] of [ComponentValidationMessage].
 */
enum class Severity {
    Info, Success, Warning, Error
}

/**
 * Special [ValidationMessage] for fritz2 headless components.
 *
 * **Important**: [path] should be generated by using the [Inspector.path]
 * in your [Validation].
 * By default, the validation fails if one or more [ComponentValidationMessage]s have
 * a [severity] of [Severity.Error]. You can override the [isError] method to change this
 * behavior.
 *
 * @param path location of the validated field in model
 * @param severity used for rendering the [ValidationMessage]
 * @param message contains the message
 * @param details optional details for extending the message
 */
open class ComponentValidationMessage(
    override val path: String,
    val severity: Severity,
    val message: String,
    val details: String? = null,
) : ValidationMessage {

    override val isError: Boolean = severity > Severity.Warning

    override fun toString(): String = buildString {
        append("{ ")
        append("\"path\": \"$path\",")
        append("\"severity\": \"$severity\",")
        append("\"message\": \"$message\",")
        append("\"details\": \"${details ?: ""}\"")
        append(" }")
    }
}

/**
 * Creates [ComponentValidationMessage] with [Severity.Info].
 *
 * @param path location of the validated field in model
 * @param message contains the message
 * @param details optional details for extending the message
 */
fun infoMessage(path: String, message: String, details: String? = null) =
    ComponentValidationMessage(path, Severity.Info, message, details)

/**
 * Creates [ComponentValidationMessage] with [Severity.Info].
 *
 * @param message contains the message
 * @param details optional details for extending the message
 */
fun <T> Inspector<T>.infoMessage(message: String, details: String? = null) =
    ComponentValidationMessage(path, Severity.Info, message, details)

/**
 * Creates [ComponentValidationMessage] with [Severity.Info].
 *
 * @param path location of the validated field in model
 * @param message contains the message
 * @param details optional details for extending the message
 */
fun successMessage(path: String, message: String, details: String? = null) =
    ComponentValidationMessage(path, Severity.Success, message, details)

/**
 * Creates [ComponentValidationMessage] with [Severity.Info].
 *
 * @param message contains the message
 * @param details optional details for extending the message
 */
fun <T> Inspector<T>.successMessage(message: String, details: String? = null) =
    ComponentValidationMessage(path, Severity.Success, message, details)

/**
 * Creates [ComponentValidationMessage] with [Severity.Warning].
 *
 * @param path location of the validated field in model
 * @param message contains the message
 * @param details optional details for extending the message
 */
fun warningMessage(path: String, message: String, details: String? = null) =
    ComponentValidationMessage(path, Severity.Warning, message, details)

/**
 * Creates [ComponentValidationMessage] with [Severity.Warning].
 *
 * @param message contains the message
 * @param details optional details for extending the message
 */
fun <T> Inspector<T>.warningMessage(message: String, details: String? = null) =
    ComponentValidationMessage(path, Severity.Warning, message, details)

/**
 * Creates [ComponentValidationMessage] with [Severity.Error].
 *
 * @param path location of the validated field in model
 * @param message contains the message
 * @param details optional details for extending the message
 */
fun errorMessage(path: String, message: String, details: String? = null) =
    ComponentValidationMessage(path, Severity.Error, message, details)

/**
 * Creates [ComponentValidationMessage] with [Severity.Error].
 *
 * @param message contains the message
 * @param details optional details for extending the message
 */
fun <T> Inspector<T>.errorMessage(message: String, details: String? = null) =
    ComponentValidationMessage(path, Severity.Error, message, details)
