package dev.fritz2.core

/**
 *  gives you a new [RootInspector] as starting point.
 */
fun <T> inspectorOf(data: T) = RootInspector(data)

/**
 * represents the data and corresponding id of certain value
 * in a deep nested model structure.
 *
 * @property data [T] representation of stored data
 * @property path [String] representation of the corresponding path in model
 */
interface Inspector<T> {
    val data: T
    val path: String

    /**
     * creates a new [Inspector] for a part of your underlying data-model
     *
     * @param lens a [Lens] describing, of which part of your data
     * model you want to have the next [Inspector]
     */
    fun <X> sub(lens: Lens<T, X>): Inspector<X>
}


/**
 * [RootInspector] is the starting point for getting your [data] and corresponding [path]s from your
 * deep nested model structure. Get this by calling the factory method [inspectorOf].
 *
 * [Inspector] is useful in validation process to know which model attribute is not valid.
 */
class RootInspector<T>(
    override val data: T
) : Inspector<T> {

    override val path: String = ""

    override fun <X> sub(lens: Lens<T, X>): SubInspector<T, T, X> =
        SubInspector(this, lens, this, lens)
}

/**
 *  [SubInspector] is the next lower [Inspector] in a deep nested model structure.
 *  It's generated by calling the [sub] function on an [Inspector].
 */
class SubInspector<R, P, T>(
    private val parent: Inspector<P>,
    private val lens: Lens<P, T>,
    val rootModelId: RootInspector<R>,
    val rootLens: Lens<R, T>
) : Inspector<T> {

    /**
     * generates the corresponding [path]
     */
    override val path: String by lazy { "${parent.path}.${lens.id}".trimEnd('.') }

    /**
     * returns the underlying [data]
     */
    override val data: T = lens.get(parent.data)

    override fun <X> sub(lens: Lens<T, X>): SubInspector<R, T, X> =
        SubInspector(this, lens, rootModelId, this.rootLens + lens)
}

/**
 * creates a [Inspector] for an element in your [Inspector]'s list.
 *
 * @param element to get the [Inspector] for
 * @param idProvider to get the id from an instance
 */
inline fun <reified T, I> RootInspector<List<T>>.sub(
    element: T,
    noinline idProvider: IdProvider<T, I>
): SubInspector<List<T>, List<T>, T> {
    val lens = lensOf(element, idProvider)
    return SubInspector(this, lens, this, lens)
}

/**
 * Performs the given [action] on each [SubInspector].
 *
 * @param idProvider to get the id from an instance
 * @param action function which gets applied to all [SubInspector]s
 */
inline fun <reified T, I> RootInspector<List<T>>.inspectEach(
    noinline idProvider: IdProvider<T, I>,
    action: (SubInspector<List<T>, List<T>, T>) -> Unit
) { this.data.onEach { element -> action(sub(element, idProvider)) } }

/**
 * creates a [Inspector] for an element in your [Inspector]'s list.
 *
 * @param index you need the [Inspector] for
 */
inline fun <reified X> RootInspector<List<X>>.sub(index: Int): SubInspector<List<X>, List<X>, X> {
    val lens = lensOf<X>(index)
    return SubInspector(this, lens, this, lens)
}

/**
 * Performs the given [action] on each [SubInspector].
 *
 * @param action function which gets applied to all [SubInspector]s
 */
inline fun <reified X> RootInspector<List<X>>.inspectEach(
    action: (SubInspector<List<X>, List<X>, X>) -> Unit
) { this.data.onEachIndexed { index, _ -> action(sub(index)) } }

/**
 * creates a [Inspector] for an element in your [Inspector]'s list.
 *
 * @param element to get the [Inspector] for
 * @param idProvider to get the id from an instance
 */
inline fun <R, P, reified T, I> SubInspector<R, P, List<T>>.sub(
    element: T,
    noinline idProvider: IdProvider<T, I>
): SubInspector<R, List<T>, T> {
    val lens = lensOf(element, idProvider)
    return SubInspector(this, lens, this.rootModelId, this.rootLens + lens)
}

/**
 * Performs the given [action] on each [SubInspector].
 *
 * @param idProvider to get the id from an instance
 * @param action function which gets applied to all [SubInspector]s
 */
inline fun <R, P, reified T, I> SubInspector<R, P, List<T>>.inspectEach(
    noinline idProvider: IdProvider<T, I>,
    action: (SubInspector<R, List<T>, T>) -> Unit
) { this.data.onEach { element -> action(sub(element, idProvider)) } }

/**
 * creates a [Inspector] for an element in your [Inspector]'s list.
 *
 * @param index of the element in your list you need the [Inspector] for
 */
inline fun <R, P, reified X> SubInspector<R, P, List<X>>.sub(index: Int): SubInspector<R, List<X>, X> {
    val lens = lensOf<X>(index)
    return SubInspector(this, lens, this.rootModelId, this.rootLens + lens)
}

/**
 * Performs the given [action] on each [SubInspector].
 *
 * @param action function which gets applied to all [SubInspector]s
 */
inline fun <R, P, reified X> SubInspector<R, P, List<X>>.inspectEach(
    action: (SubInspector<R, List<X>, X>) -> Unit
) { this.data.onEachIndexed { index, _ -> action(sub(index)) } }
