/*
 * Decompiled with CFR 0.152.
 */
package dev.dsf.tools.generator;

import com.google.common.collect.Streams;
import de.rwh.utils.crypto.CertificateAuthority;
import de.rwh.utils.crypto.CertificateHelper;
import de.rwh.utils.crypto.CertificationRequestBuilder;
import de.rwh.utils.crypto.io.CsrIo;
import de.rwh.utils.crypto.io.PemIo;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPublicKeySpec;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x500.style.IETFUtils;
import org.bouncycastle.cert.jcajce.JcaX509CertificateHolder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertificateGenerator {
    private static final Logger logger = LoggerFactory.getLogger(CertificateGenerator.class);
    private static final char[] CERT_PASSWORD = "password".toCharArray();
    private static final String[] SERVER_COMMON_NAMES = new String[]{"localhost", "keycloak"};
    private static final String[] CLIENT_COMMON_NAMES = new String[]{"ttp-client", "dic1-client", "dic2-client", "dic3-client", "test-client", "Webbrowser Test User"};
    private static final Map<String, List<String>> DNS_NAMES = Map.of("localhost", Arrays.asList("localhost", "host.docker.internal", "fhir", "bpe", "ttp", "dic1", "dic2", "dic3"));
    private static final BouncyCastleProvider PROVIDER = new BouncyCastleProvider();
    private CertificateAuthority ca;
    private Map<String, CertificateFiles> serverCertificateFilesByCommonName;
    private Map<String, CertificateFiles> clientCertificateFilesByCommonName;

    public void generateCertificates() {
        this.ca = this.initCA();
        this.serverCertificateFilesByCommonName = Arrays.stream(SERVER_COMMON_NAMES).map(commonName -> this.createCert(CertificateType.SERVER, (String)commonName, DNS_NAMES.getOrDefault(commonName, Collections.singletonList(commonName)))).collect(Collectors.toMap(CertificateFiles::getCommonName, Function.identity()));
        this.clientCertificateFilesByCommonName = Arrays.stream(CLIENT_COMMON_NAMES).map(commonName -> this.createCert(CertificateType.CLIENT, (String)commonName, Collections.emptyList())).collect(Collectors.toMap(CertificateFiles::getCommonName, Function.identity()));
        this.writeThumbprints();
    }

    public Map<String, CertificateFiles> getServerCertificateFilesByCommonName() {
        return this.serverCertificateFilesByCommonName != null ? Collections.unmodifiableMap(this.serverCertificateFilesByCommonName) : Collections.emptyMap();
    }

    public Map<String, CertificateFiles> getClientCertificateFilesByCommonName() {
        return this.clientCertificateFilesByCommonName != null ? Collections.unmodifiableMap(this.clientCertificateFilesByCommonName) : Collections.emptyMap();
    }

    public CertificateAuthority initCA() {
        Path caCertFile = this.createFolderIfNotExists(Paths.get("cert/ca/testca_certificate.pem", new String[0]));
        Path caPrivateKeyFile = this.createFolderIfNotExists(Paths.get("cert/ca/testca_private-key.pem", new String[0]));
        if (Files.isReadable(caCertFile) && Files.isReadable(caPrivateKeyFile)) {
            logger.info("Initializing CA from cert file: {}, private key {}", (Object)caCertFile.toString(), (Object)caPrivateKeyFile.toString());
            X509Certificate caCertificate = this.readCertificate(caCertFile);
            PrivateKey caPrivateKey = this.readPrivatekey(caPrivateKeyFile);
            return CertificateAuthority.CertificateAuthorityBuilder.create((X509Certificate)caCertificate, (PrivateKey)caPrivateKey).initialize();
        }
        logger.info("Initializing CA with new cert file: {}, private key {}", (Object)caCertFile.toString(), (Object)caPrivateKeyFile.toString());
        CertificateAuthority ca = CertificateAuthority.CertificateAuthorityBuilder.create((String)"DE", null, null, null, null, (String)"Test").initialize();
        this.writeCertificate(caCertFile, ca.getCertificate());
        this.writePrivateKeyEncrypted(caPrivateKeyFile, ca.getCaKeyPair().getPrivate());
        return ca;
    }

    private void writePrivateKeyEncrypted(Path privateKeyFile, PrivateKey privateKey) {
        try {
            PemIo.writeAes128EncryptedPrivateKeyToPkcs8((BouncyCastleProvider)PROVIDER, (Path)privateKeyFile, (PrivateKey)privateKey, (char[])CERT_PASSWORD);
        }
        catch (IOException | OperatorCreationException e) {
            logger.error("Error while writing encrypted private-key to {}", (Object)privateKeyFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private void writePrivateKeyNotEncrypted(Path privateKeyFile, PrivateKey privateKey) {
        try {
            PemIo.writeNotEncryptedPrivateKeyToPkcs8((BouncyCastleProvider)PROVIDER, (Path)privateKeyFile, (PrivateKey)privateKey);
        }
        catch (IOException | OperatorCreationException e) {
            logger.error("Error while writing not-encrypted private-key to {}", (Object)privateKeyFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private void writeCertificate(Path certificateFile, X509Certificate certificate) {
        try {
            PemIo.writeX509CertificateToPem((X509Certificate)certificate, (Path)certificateFile);
        }
        catch (IOException | IllegalStateException | CertificateEncodingException e) {
            logger.error("Error while writing certificate to {}", (Object)certificateFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private PrivateKey readPrivatekey(Path privateKeyFile) {
        try {
            return PemIo.readPrivateKeyFromPem((BouncyCastleProvider)PROVIDER, (Path)privateKeyFile, (char[])CERT_PASSWORD);
        }
        catch (IOException | PKCSException e) {
            logger.error("Error while reading private-key from {}", (Object)privateKeyFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private X509Certificate readCertificate(Path certFile) {
        try {
            return PemIo.readX509CertificateFromPem((Path)certFile);
        }
        catch (IOException | CertificateException e) {
            logger.error("Error while reading certificate from {}", (Object)certFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    public void writeThumbprints() {
        Path thumbprintsFile = Paths.get("cert", "thumbprints.txt");
        Stream<String> certificates = Streams.concat((Stream[])new Stream[]{this.serverCertificateFilesByCommonName.values().stream(), this.clientCertificateFilesByCommonName.values().stream()}).sorted(Comparator.comparing(CertificateFiles::getCommonName)).map(c -> c.commonName + "\n\t" + c.getCertificateSha512ThumbprintHex() + " (SHA-512)\n");
        try {
            logger.info("Writing certificate thumbprints file to {}", (Object)thumbprintsFile.toString());
            Files.write(thumbprintsFile, () -> certificates.iterator(), StandardCharsets.UTF_8, new OpenOption[0]);
        }
        catch (IOException e) {
            logger.error("Error while writing certificate thumbprints file to {}", (Object)thumbprintsFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    public CertificateFiles createCert(CertificateType certificateType, String commonName, List<String> dnsNames) {
        Path privateKeyFile = this.createFolderIfNotExists(this.getPrivateKeyPath(commonName));
        KeyPair keyPair = this.createOrReadKeyPair(privateKeyFile, commonName);
        Path certificateRequestFile = this.createFolderIfNotExists(this.getCertReqPath(commonName));
        JcaPKCS10CertificationRequest certificateRequest = this.createOrReadCertificateRequest(certificateRequestFile, certificateType, keyPair, commonName, dnsNames);
        Path certificatePemFile = this.createFolderIfNotExists(this.getCertPemPath(commonName));
        X509Certificate certificate = this.signOrReadCertificate(certificatePemFile, certificateRequest, commonName, certificateType);
        return new CertificateFiles(commonName, keyPair, certificate, this.calculateSha512CertificateThumbprint(certificate));
    }

    private X509Certificate signOrReadCertificate(Path certificateFile, JcaPKCS10CertificationRequest certificateRequest, String commonName, CertificateType certificateType) {
        if (Files.isReadable(certificateFile)) {
            logger.info("Reading certificate (pem) from {} [{}]", (Object)certificateFile.toString(), (Object)commonName);
            return this.readCertificate(certificateFile);
        }
        logger.info("Signing {} certificate [{}]", (Object)certificateType.toString().toLowerCase(), (Object)commonName);
        X509Certificate certificate = this.signCertificateRequest(certificateRequest, certificateType);
        logger.info("Saving certificate (pem) to {} [{}]", (Object)certificateFile.toString(), (Object)commonName);
        this.writeCertificate(certificateFile, certificate);
        return certificate;
    }

    private X509Certificate signCertificateRequest(JcaPKCS10CertificationRequest certificateRequest, CertificateType certificateType) {
        try {
            return switch (certificateType) {
                default -> throw new IncompatibleClassChangeError();
                case CertificateType.CLIENT -> this.ca.signWebClientCertificate(certificateRequest);
                case CertificateType.SERVER -> this.ca.signWebServerCertificate(certificateRequest);
            };
        }
        catch (IOException | IllegalStateException | InvalidKeyException | NoSuchAlgorithmException | CertificateException | InvalidKeySpecException | OperatorCreationException e) {
            logger.error("Error while signing {} certificate", (Object)certificateType.toString().toLowerCase(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private JcaPKCS10CertificationRequest createOrReadCertificateRequest(Path certificateRequestFile, CertificateType certificateType, KeyPair keyPair, String commonName, List<String> dnsNames) {
        if (!dnsNames.contains(commonName) && CertificateType.SERVER.equals((Object)certificateType)) {
            throw new IllegalArgumentException("dnsNames must contain commonName if certificateType is SERVER");
        }
        if (Files.isReadable(certificateRequestFile)) {
            logger.info("Reading certificate request (csr) from {} [{}]", (Object)certificateRequestFile.toString(), (Object)commonName);
            return this.readCertificateRequest(certificateRequestFile);
        }
        X500Name subject = CertificationRequestBuilder.createSubject((String)"DE", null, null, null, null, (String)commonName);
        JcaPKCS10CertificationRequest certificateRequest = this.createCertificateRequest(certificateType, subject, keyPair, dnsNames);
        logger.info("Saving certificate request (csr) to {} [{}]", (Object)certificateRequestFile.toString(), (Object)commonName);
        this.writeCertificateRequest(certificateRequestFile, certificateRequest);
        return certificateRequest;
    }

    private JcaPKCS10CertificationRequest createCertificateRequest(CertificateType certificateType, X500Name subject, KeyPair keyPair, List<String> dnsNames) {
        try {
            return switch (certificateType) {
                default -> throw new IncompatibleClassChangeError();
                case CertificateType.CLIENT -> CertificationRequestBuilder.createClientCertificationRequest((X500Name)subject, (KeyPair)keyPair);
                case CertificateType.SERVER -> CertificationRequestBuilder.createServerCertificationRequest((X500Name)subject, (KeyPair)keyPair, null, dnsNames);
            };
        }
        catch (IOException | IllegalStateException | NoSuchAlgorithmException | OperatorCreationException e) {
            logger.error("Error while creating certificate-request", e);
            throw new RuntimeException(e);
        }
    }

    private void writeCertificateRequest(Path certificateRequestFile, JcaPKCS10CertificationRequest certificateRequest) {
        try {
            CsrIo.writeJcaPKCS10CertificationRequestToCsr((JcaPKCS10CertificationRequest)certificateRequest, (Path)certificateRequestFile);
        }
        catch (IOException e) {
            logger.error("Error while reading certificate-request from {}", (Object)certificateRequestFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private JcaPKCS10CertificationRequest readCertificateRequest(Path certificateRequestFile) {
        try {
            return CsrIo.readJcaPKCS10CertificationRequestFromCsr((Path)certificateRequestFile);
        }
        catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            logger.error("Error while reading certificate-request from {}", (Object)certificateRequestFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private KeyPair createOrReadKeyPair(Path privateKeyFile, String commonName) {
        if (Files.isReadable(privateKeyFile)) {
            logger.info("Reading private-key from {} [{}]", (Object)privateKeyFile.toString(), (Object)commonName);
            PrivateKey privateKey = this.readPrivatekey(privateKeyFile);
            PublicKey publicKey = this.createPublicKey(privateKey, privateKeyFile, commonName);
            return new KeyPair(publicKey, privateKey);
        }
        logger.info("Generating 4096 bit key pair [{}]", (Object)commonName);
        KeyPair keyPair = this.createKeyPair();
        logger.info("Saving private-key to {} [{}]", (Object)privateKeyFile.toString(), (Object)commonName);
        this.writePrivateKeyEncrypted(privateKeyFile, keyPair.getPrivate());
        return keyPair;
    }

    private PublicKey createPublicKey(PrivateKey privateKey, Path privateKeyFile, String commonName) {
        logger.debug("Generating public-key from private-key [{}]", (Object)commonName);
        if ("RSA".equals(privateKey.getAlgorithm()) && privateKey instanceof RSAPrivateCrtKey) {
            RSAPrivateCrtKey rsaPrivateKey = (RSAPrivateCrtKey)privateKey;
            RSAPublicKeySpec publicKeySpec = new RSAPublicKeySpec(rsaPrivateKey.getModulus(), rsaPrivateKey.getPublicExponent());
            try {
                KeyFactory factory = KeyFactory.getInstance("RSA");
                return factory.generatePublic(publicKeySpec);
            }
            catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                throw new RuntimeException("Error while generating public key from private key modules and public exponent", e);
            }
        }
        throw new RuntimeException("Error while generating public key: private key for " + commonName + " at " + privateKeyFile + " not a RSA private crt key");
    }

    private KeyPair createKeyPair() {
        try {
            return CertificationRequestBuilder.createRsaKeyPair4096Bit();
        }
        catch (NoSuchAlgorithmException e) {
            logger.error("Error while creating RSA key pair", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    private Path createFolderIfNotExists(Path file) {
        try {
            Files.createDirectories(file.getParent(), new FileAttribute[0]);
        }
        catch (IOException e) {
            logger.error("Error while creating directories {}", (Object)file.getParent().toString(), (Object)e);
            throw new RuntimeException(e);
        }
        return file;
    }

    private Path getCertReqPath(String commonName) {
        commonName = commonName.replaceAll("\\s+", "_");
        return Paths.get("cert", commonName, commonName + "_certificate.csr");
    }

    private Path getCertP12Path(String commonName) {
        commonName = commonName.replaceAll("\\s+", "_");
        return Paths.get("cert", commonName, commonName + "_certificate.p12");
    }

    private Path getCertPemPath(String commonName) {
        commonName = commonName.replaceAll("\\s+", "_");
        return Paths.get("cert", commonName, commonName + "_certificate.pem");
    }

    private Path getPrivateKeyPath(String commonName) {
        commonName = commonName.replaceAll("\\s+", "_");
        return Paths.get("cert", commonName, commonName + "_private-key.pem");
    }

    private byte[] calculateSha512CertificateThumbprint(X509Certificate certificate) {
        try {
            return MessageDigest.getInstance("SHA-512").digest(certificate.getEncoded());
        }
        catch (NoSuchAlgorithmException | CertificateEncodingException e) {
            logger.error("Error while calculating SHA-512 certificate thumbprint", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    public void copyJavaTestCertificates() {
        X509Certificate testCaCertificate = this.ca.getCertificate();
        Path bpeCaCertFile = Paths.get("../../dsf-bpe/dsf-bpe-server-jetty/target/testca_certificate.pem", new String[0]);
        logger.info("Copying Test CA certificate file to {}", (Object)bpeCaCertFile.toString());
        this.writeCertificate(bpeCaCertFile, testCaCertificate);
        Path fhirCacertFile = Paths.get("../../dsf-fhir/dsf-fhir-server-jetty/target/testca_certificate.pem", new String[0]);
        logger.info("Copying Test CA certificate file to {}", (Object)fhirCacertFile.toString());
        this.writeCertificate(fhirCacertFile, testCaCertificate);
        CertificateFiles localhost = this.serverCertificateFilesByCommonName.get("localhost");
        Path bpeCertificateFile = Paths.get("../../dsf-bpe/dsf-bpe-server-jetty/target/localhost_certificate.pem", new String[0]);
        logger.info("Copying localhost certificate file to {}", (Object)bpeCertificateFile.toString());
        this.writeCertificate(bpeCertificateFile, localhost.certificate);
        Path bpeCertificatePrivateKeyFile = Paths.get("../../dsf-bpe/dsf-bpe-server-jetty/target/localhost_private-key.pem", new String[0]);
        logger.info("Copying localhost certificate private-key file to {}", (Object)bpeCertificateFile.toString());
        this.writePrivateKeyEncrypted(bpeCertificatePrivateKeyFile, localhost.keyPair.getPrivate());
        Path fhirCertificateFile = Paths.get("../../dsf-fhir/dsf-fhir-server-jetty/target/localhost_certificate.pem", new String[0]);
        logger.info("Copying localhost certificate file to {}", (Object)fhirCertificateFile.toString());
        this.writeCertificate(fhirCertificateFile, localhost.certificate);
        Path fhirCertificatePrivateKeyFile = Paths.get("../../dsf-fhir/dsf-fhir-server-jetty/target/localhost_private-key.pem", new String[0]);
        logger.info("Copying localhost certificate private-key file to {}", (Object)fhirCertificateFile.toString());
        this.writePrivateKeyEncrypted(fhirCertificatePrivateKeyFile, localhost.keyPair.getPrivate());
        CertificateFiles testClient = this.clientCertificateFilesByCommonName.get("test-client");
        Path bpeClientCertificateFile = Paths.get("../../dsf-bpe/dsf-bpe-server-jetty/target/test-client_certificate.pem", new String[0]);
        logger.info("Copying test-client certificate file to {}", (Object)bpeClientCertificateFile);
        this.writeCertificate(bpeClientCertificateFile, testClient.certificate);
        Path bpeClientPrivateKeyFile = Paths.get("../../dsf-bpe/dsf-bpe-server-jetty/target/test-client_private-key.pem", new String[0]);
        logger.info("Copying test-client certificate private-key file to {}", (Object)bpeClientPrivateKeyFile);
        this.writePrivateKeyEncrypted(bpeClientPrivateKeyFile, testClient.keyPair.getPrivate());
        Path fhirClientCertificateFile = Paths.get("../../dsf-fhir/dsf-fhir-server-jetty/target/test-client_certificate.pem", new String[0]);
        logger.info("Copying test-client certificate file to {}", (Object)fhirClientCertificateFile);
        this.writeCertificate(fhirClientCertificateFile, testClient.certificate);
        Path fhirClientPrivateKeyFile = Paths.get("../../dsf-fhir/dsf-fhir-server-jetty/target/test-client_private-key.pem", new String[0]);
        logger.info("Copying test-client certificate private-key file to {}", (Object)fhirClientPrivateKeyFile);
        this.writePrivateKeyEncrypted(fhirClientPrivateKeyFile, testClient.keyPair.getPrivate());
    }

    public void copyDockerTestCertificates() {
        this.copyProxyFiles("dsf-docker-test-setup", "localhost");
        this.copyClientCertFiles("../../dsf-docker-test-setup/bpe/secrets/", "../../dsf-docker-test-setup/fhir/secrets/", "test-client");
    }

    private void copyProxyFiles(String dockerTestFolder, String commonName) {
        X509Certificate testCaCertificate = this.ca.getCertificate();
        CertificateFiles serverCertFiles = this.serverCertificateFilesByCommonName.get(commonName);
        Path baseFolder = Paths.get("../../", dockerTestFolder);
        Path bpeCertificateFile = baseFolder.resolve("bpe/secrets/server_certificate.pem");
        logger.info("Copying {} certificate pem file to {}", (Object)commonName, (Object)bpeCertificateFile);
        this.writeCertificate(bpeCertificateFile, serverCertFiles.getCertificate());
        Path bpeCertificatePrivateKeyFile = baseFolder.resolve("bpe/secrets/server_certificate_private_key.pem");
        logger.info("Copying {} private-key file to {}", (Object)commonName, (Object)bpeCertificatePrivateKeyFile);
        this.writePrivateKeyNotEncrypted(bpeCertificatePrivateKeyFile, serverCertFiles.keyPair.getPrivate());
        Path bpeTestCaCertificate = baseFolder.resolve("bpe/secrets/testca_certificate.pem");
        logger.info("Copying Test CA certificate file to {}", (Object)bpeTestCaCertificate.toString());
        this.writeCertificate(bpeTestCaCertificate, testCaCertificate);
        Path fhirCertificateFile = baseFolder.resolve("fhir/secrets/server_certificate.pem");
        logger.info("Copying {} certificate pem file to {}", (Object)commonName, (Object)fhirCertificateFile);
        this.writeCertificate(fhirCertificateFile, serverCertFiles.getCertificate());
        Path fhirCertificatePrivateKeyFile = baseFolder.resolve("fhir/secrets/server_certificate_private_key.pem");
        logger.info("Copying {} private-key file to {}", (Object)commonName, (Object)fhirCertificatePrivateKeyFile);
        this.writePrivateKeyNotEncrypted(fhirCertificatePrivateKeyFile, serverCertFiles.keyPair.getPrivate());
        Path fhirTestCaCertificate = baseFolder.resolve("fhir/secrets/testca_certificate.pem");
        logger.info("Copying Test CA certificate file to {}", (Object)fhirTestCaCertificate.toString());
        this.writeCertificate(fhirTestCaCertificate, testCaCertificate);
    }

    private void copyClientCertFiles(String bpeConfFolder, String fhirConfFolder, String commonName) {
        CertificateFiles clientCertFiles = this.clientCertificateFilesByCommonName.get(commonName);
        Path bpeClientCertificateFile = Paths.get(bpeConfFolder, "client_certificate.pem");
        logger.info("Copying {} certificate certificate file to {}", (Object)commonName, (Object)bpeClientCertificateFile);
        this.writeCertificate(bpeClientCertificateFile, clientCertFiles.certificate);
        Path bpeClientPrivateKeyFile = Paths.get(bpeConfFolder, "client_certificate_private_key.pem");
        logger.info("Copying {} certificate private-key file to {}", (Object)commonName, (Object)bpeClientPrivateKeyFile);
        this.writePrivateKeyEncrypted(bpeClientPrivateKeyFile, clientCertFiles.keyPair.getPrivate());
        Path fhirClientCertificateFile = Paths.get(fhirConfFolder, "client_certificate.pem");
        logger.info("Copying {} certificate certificate file to {}", (Object)commonName, (Object)fhirClientCertificateFile);
        this.writeCertificate(fhirClientCertificateFile, clientCertFiles.certificate);
        Path fhirClientPrivateKeyFile = Paths.get(fhirConfFolder, "client_certificate_private_key.pem");
        logger.info("Copying {} certificate private-key file to {}", (Object)commonName, (Object)fhirClientPrivateKeyFile);
        this.writePrivateKeyEncrypted(fhirClientPrivateKeyFile, clientCertFiles.keyPair.getPrivate());
    }

    public void copyDockerTest3DicTtpCertificates() {
        Path baseFolder = Paths.get("../../dsf-docker-test-setup-3dic-ttp/secrets/", new String[0]);
        X509Certificate testCaCertificate = this.ca.getCertificate();
        Path testCaCertificateFile = baseFolder.resolve("proxy_trusted_client_cas.pem");
        logger.info("Copying Test CA certificate file to {}", (Object)testCaCertificateFile.toString());
        this.writeCertificate(testCaCertificateFile, testCaCertificate);
        CertificateFiles localhost = this.serverCertificateFilesByCommonName.get("localhost");
        Path localhostCertificateAndCa = baseFolder.resolve("proxy_certificate_and_int_cas.pem");
        logger.info("Writing localhost certificate and CA certificate to {}", (Object)testCaCertificateFile.toString());
        this.writeCertificates(localhostCertificateAndCa, localhost.getCertificate());
        Path localhostCertificatePrivateKey = baseFolder.resolve("proxy_certificate_private_key.pem");
        logger.info("Copying localhost private-key file to {}", (Object)localhostCertificatePrivateKey);
        this.writePrivateKeyNotEncrypted(localhostCertificatePrivateKey, localhost.keyPair.getPrivate());
        List<String> commonNames = Arrays.asList("dic1", "dic2", "dic3", "ttp");
        commonNames.forEach(cn -> this.copyDockerTest3DicTtpClientCertFiles("../../dsf-docker-test-setup-3dic-ttp/secrets/", cn + "-client"));
        Path fhirCacertFile = baseFolder.resolve("app_testca_certificate.pem");
        logger.info("Copying Test CA certificate file to {}", (Object)fhirCacertFile.toString());
        this.writeCertificate(fhirCacertFile, testCaCertificate);
        CertificateFiles keycloak = this.serverCertificateFilesByCommonName.get("keycloak");
        Path keycloakCertificateAndCa = baseFolder.resolve("keycloak_certificate_and_int_cas.pem");
        logger.info("Writing keycloak certificate and CA certificate to {}", (Object)testCaCertificateFile.toString());
        this.writeCertificates(keycloakCertificateAndCa, keycloak.getCertificate());
        Path keycloakCertificatePrivateKey = baseFolder.resolve("keycloak_certificate_private_key.pem");
        logger.info("Copying keycloak private-key file to {}", (Object)keycloakCertificatePrivateKey);
        this.writePrivateKeyNotEncrypted(keycloakCertificatePrivateKey, keycloak.keyPair.getPrivate());
        Path keycloakTrustStoreFile = baseFolder.resolve("keycloak_trust_store.jks");
        logger.info("Copying Test CA certificate as trust store file to {}", (Object)keycloakTrustStoreFile.toString());
        KeyStore trustStore = this.createJksKeyStore(this.getCommonName(this.ca.getCertificate()), testCaCertificate);
        this.writeKeyStore(keycloakTrustStoreFile, trustStore);
    }

    private String getCommonName(X509Certificate certificate) {
        try {
            return IETFUtils.valueToString((ASN1Encodable)new JcaX509CertificateHolder(certificate).getSubject().getRDNs(BCStyle.CN)[0].getFirst().getValue());
        }
        catch (CertificateEncodingException e) {
            logger.error("Error unable to extract common-name from certificate", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    private void copyDockerTest3DicTtpClientCertFiles(String folder, String commonName) {
        CertificateFiles clientCertFiles = this.clientCertificateFilesByCommonName.get(commonName);
        Path bpeClientCertificateFile = Paths.get(folder, "app_" + commonName + "_certificate.pem");
        logger.info("Copying {} certificate certificate file to {}", (Object)commonName, (Object)bpeClientCertificateFile);
        this.writeCertificate(bpeClientCertificateFile, clientCertFiles.certificate);
        Path bpeClientPrivateKeyFile = Paths.get(folder, "app_" + commonName + "_private-key.pem");
        logger.info("Copying {} certificate private-key file to {}", (Object)commonName, (Object)bpeClientPrivateKeyFile);
        this.writePrivateKeyEncrypted(bpeClientPrivateKeyFile, clientCertFiles.keyPair.getPrivate());
    }

    private void writeCertificates(Path certificateFile, X509Certificate ... certificates) {
        try {
            StringBuilder b = new StringBuilder();
            for (X509Certificate cert : certificates) {
                b.append("subject= ");
                b.append(cert.getSubjectX500Principal().getName());
                b.append("\n");
                b.append(PemIo.writeX509Certificate((X509Certificate)cert));
            }
            Files.writeString(certificateFile, (CharSequence)b.toString(), new OpenOption[0]);
        }
        catch (IOException | IllegalStateException | CertificateEncodingException e) {
            logger.error("Error while writing certificate to {}", (Object)certificateFile.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    private KeyStore createJksKeyStore(String commonName, X509Certificate certificate) {
        try {
            KeyStore keyStore = KeyStore.getInstance("jks");
            keyStore.load(null, null);
            keyStore.setCertificateEntry(commonName, certificate);
            return keyStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            logger.error("Error while creating jks key-store", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    private KeyStore createP12KeyStore(PrivateKey privateKey, String commonName, X509Certificate certificate) {
        try {
            return CertificateHelper.toPkcs12KeyStore((PrivateKey)privateKey, (Certificate[])new Certificate[]{certificate, this.ca.getCertificate()}, (String)commonName, (char[])CERT_PASSWORD);
        }
        catch (IOException | IllegalStateException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            logger.error("Error while creating P12 key-store", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    private void writeKeyStore(Path file, KeyStore keyStore) {
        try (OutputStream stream = Files.newOutputStream(file, new OpenOption[0]);){
            keyStore.store(stream, CERT_PASSWORD);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            logger.error("Error while writing keystore file to {}", (Object)file.toString(), (Object)e);
            throw new RuntimeException(e);
        }
    }

    public Path createP12(CertificateFiles files) {
        Path certP12Path = this.getCertP12Path(files.commonName);
        logger.info("Saving certificate (p21) to {}, password '{}' [{}]", new Object[]{certP12Path.toString(), String.valueOf(CERT_PASSWORD), files.commonName});
        KeyStore p12KeyStore = this.createP12KeyStore(files.keyPair.getPrivate(), files.commonName, files.certificate);
        this.writeKeyStore(certP12Path, p12KeyStore);
        return certP12Path;
    }

    public static void main(String[] args) {
        CertificateAuthority.registerBouncyCastleProvider();
        new CertificateGenerator().generateCertificates();
    }

    private static enum CertificateType {
        CLIENT,
        SERVER;

    }

    public static final class CertificateFiles {
        private final String commonName;
        private final KeyPair keyPair;
        private final X509Certificate certificate;
        private final byte[] certificateSha512Thumbprint;

        CertificateFiles(String commonName, KeyPair keyPair, X509Certificate certificate, byte[] certificateSha512Thumbprint) {
            this.commonName = commonName;
            this.keyPair = keyPair;
            this.certificate = certificate;
            this.certificateSha512Thumbprint = certificateSha512Thumbprint;
        }

        public String getCommonName() {
            return this.commonName;
        }

        public X509Certificate getCertificate() {
            return this.certificate;
        }

        public String getCertificateSha512ThumbprintHex() {
            return Hex.encodeHexString((byte[])this.certificateSha512Thumbprint);
        }
    }
}

