package dev.dsf.tools.generator;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import dev.dsf.tools.generator.CertificateGenerator.CertificateFiles;

public class ConfigGenerator
{
	private static final Logger logger = LoggerFactory.getLogger(ConfigGenerator.class);

	private static final String P_KEY_ROLE_CONFIG = "dev.dsf.fhir.server.roleConfig";

	private Properties javaTestFhirConfigProperties;

	private Properties readProperties(Path propertiesFile)
	{
		@SuppressWarnings("serial")
		Properties properties = new Properties()
		{
			// making sure entries are sorted when storing properties
			@Override
			public Set<java.util.Map.Entry<Object, Object>> entrySet()
			{
				return Collections.synchronizedSet(
						super.entrySet().stream().sorted(Comparator.comparing(e -> e.getKey().toString()))
								.collect(Collectors.toCollection(LinkedHashSet::new)));
			}
		};
		try (InputStream in = Files.newInputStream(propertiesFile);
				InputStreamReader reader = new InputStreamReader(in, StandardCharsets.UTF_8))
		{
			properties.load(reader);
		}
		catch (IOException e)
		{
			logger.error("Error while reading properties from " + propertiesFile.toString(), e);
			throw new RuntimeException(e);
		}
		return properties;
	}

	private void writeProperties(Path propertiesFiles, Properties properties)
	{
		try (OutputStream out = Files.newOutputStream(propertiesFiles);
				OutputStreamWriter writer = new OutputStreamWriter(out, StandardCharsets.UTF_8))
		{
			properties.store(writer, "Generated by test-data-generator");
		}
		catch (IOException e)
		{
			logger.error("Error while writing properties to " + propertiesFiles.toString(), e);
			throw new RuntimeException(e);
		}
	}

	public void modifyJavaTestFhirConfigProperties(Map<String, CertificateFiles> clientCertificateFilesByCommonName)
	{
		CertificateFiles webbrowserTestUser = clientCertificateFilesByCommonName.get("Webbrowser Test User");

		Path javaTestFhirConfigTemplateFile = Paths
				.get("src/main/resources/config-templates/java-test-fhir-config.properties");
		javaTestFhirConfigProperties = readProperties(javaTestFhirConfigTemplateFile);
		javaTestFhirConfigProperties.setProperty(P_KEY_ROLE_CONFIG, String.format("""
				- webbrowser_test_user:
				    thumbprint: %s
				    token-role: admin
				    dsf-role:
				      - CREATE
				      - READ
				      - UPDATE
				      - DELETE
				      - SEARCH
				      - HISTORY
				      - PERMANENT_DELETE
				    practitioner-role:
				      - http://dsf.dev/fhir/CodeSystem/practitioner-role|DSF_ADMIN
				""", webbrowserTestUser.getCertificateSha512ThumbprintHex()));

		writeProperties(Paths.get("config/java-test-fhir-config.properties"), javaTestFhirConfigProperties);
	}

	public void copyJavaTestFhirConfigProperties()
	{
		Path javaTestConfigPropertiesFile = Paths.get("../../dsf-fhir/dsf-fhir-server-jetty/conf/config.properties");
		logger.info("Copying config.properties to {}", javaTestConfigPropertiesFile);
		writeProperties(javaTestConfigPropertiesFile, javaTestFhirConfigProperties);
	}
}
