/*
 * Decompiled with CFR 0.152.
 */
package dev.dsf.fhir.service;

import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.context.support.ConceptValidationOptions;
import ca.uhn.fhir.context.support.IValidationSupport;
import ca.uhn.fhir.context.support.ValidationSupportContext;
import ca.uhn.fhir.context.support.ValueSetExpansionOptions;
import dev.dsf.fhir.event.Event;
import dev.dsf.fhir.event.EventHandler;
import dev.dsf.fhir.event.ResourceCreatedEvent;
import dev.dsf.fhir.event.ResourceDeletedEvent;
import dev.dsf.fhir.event.ResourceUpdatedEvent;
import java.lang.ref.SoftReference;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.hl7.fhir.instance.model.api.IBaseResource;
import org.hl7.fhir.r4.model.CodeSystem;
import org.hl7.fhir.r4.model.Resource;
import org.hl7.fhir.r4.model.StructureDefinition;
import org.hl7.fhir.r4.model.ValueSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ValidationSupportWithCache
implements IValidationSupport,
EventHandler {
    private static final Logger logger = LoggerFactory.getLogger(ValidationSupportWithCache.class);
    private static final Pattern UUID_PATTERN = Pattern.compile("[a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}");
    private final FhirContext context;
    private final IValidationSupport delegate;
    private final AtomicBoolean fetchAllStructureDefinitionsDone = new AtomicBoolean();
    private final AtomicBoolean fetchAllConformanceResourcesDone = new AtomicBoolean();
    private final ConcurrentMap<String, CacheEntry<StructureDefinition>> structureDefinitions = new ConcurrentHashMap<String, CacheEntry<StructureDefinition>>();
    private final ConcurrentMap<String, CacheEntry<CodeSystem>> codeSystems = new ConcurrentHashMap<String, CacheEntry<CodeSystem>>();
    private final ConcurrentMap<String, CacheEntry<ValueSet>> valueSets = new ConcurrentHashMap<String, CacheEntry<ValueSet>>();
    private final ConcurrentMap<String, String> urlAndVersionsById = new ConcurrentHashMap<String, String>();

    public ValidationSupportWithCache(FhirContext context, IValidationSupport delegate) {
        this.context = context;
        this.delegate = delegate;
    }

    public ValidationSupportWithCache populateCache(List<IBaseResource> cacheValues) {
        logger.trace("populating cache");
        cacheValues.stream().filter(r -> r instanceof Resource).map(r -> (Resource)r).forEach(this::add);
        this.fetchAllConformanceResourcesDone.set(true);
        this.fetchAllStructureDefinitionsDone.set(true);
        return this;
    }

    public FhirContext getFhirContext() {
        return this.context;
    }

    @Override
    public void handleEvent(Event event) {
        if (event == null) {
            return;
        }
        logger.trace("handling event {}", (Object)event.getClass().getSimpleName());
        if (event instanceof ResourceCreatedEvent && this.resourceSupported(event.getResource())) {
            this.add(event.getResource());
        } else if (event instanceof ResourceDeletedEvent && this.resourceSupported(event.getResourceType(), event.getId())) {
            this.remove(event.getResourceType(), event.getId());
        } else if (event instanceof ResourceUpdatedEvent && this.resourceSupported(event.getResource())) {
            this.update(event.getResource());
        }
    }

    private boolean resourceSupported(Resource resource) {
        return resource != null && (resource instanceof CodeSystem || resource instanceof StructureDefinition || resource instanceof ValueSet);
    }

    private boolean resourceSupported(Class<? extends Resource> type, String resourceId) {
        return this.urlAndVersionsById.containsKey(resourceId) && (CodeSystem.class.equals(type) || StructureDefinition.class.equals(type) || ValueSet.class.equals(type));
    }

    private void add(Resource resource) {
        if (resource instanceof CodeSystem) {
            CodeSystem c = (CodeSystem)resource;
            this.doAdd(c, this.codeSystems, CodeSystem::getUrl, CodeSystem::getVersion, url -> (CodeSystem)this.delegate.fetchCodeSystem(url));
        } else if (resource instanceof StructureDefinition) {
            StructureDefinition s = (StructureDefinition)resource;
            this.doAdd(s, this.structureDefinitions, StructureDefinition::getUrl, StructureDefinition::getVersion, url -> (StructureDefinition)this.delegate.fetchStructureDefinition(url));
        } else if (resource instanceof ValueSet) {
            ValueSet v = (ValueSet)resource;
            this.doAdd(v, this.valueSets, ValueSet::getUrl, ValueSet::getVersion, url -> (ValueSet)this.delegate.fetchValueSet(url));
        }
    }

    private <R extends Resource> void doAdd(R resource, ConcurrentMap<String, CacheEntry<R>> cache, Function<R, String> toUrl, Function<R, String> toVersion, Function<String, R> fetch) {
        String url = toUrl.apply(resource);
        String version = toVersion.apply(resource);
        cache.put(url, new CacheEntry<Resource>(resource, () -> (Resource)fetch.apply(url)));
        if (version != null) {
            String urlAndVersion = url + "|" + version;
            cache.put(urlAndVersion, new CacheEntry<Resource>(resource, () -> (Resource)fetch.apply(urlAndVersion)));
        }
        if (resource.hasIdElement() && resource.getIdElement().hasIdPart() && UUID_PATTERN.matcher(resource.getIdElement().getIdPart()).matches()) {
            this.urlAndVersionsById.put(resource.getIdElement().getIdPart(), url + "|" + version);
        }
    }

    private void update(Resource resource) {
        this.remove(resource);
        this.add(resource);
    }

    private void remove(Resource resource) {
        if (resource instanceof CodeSystem) {
            CodeSystem c = (CodeSystem)resource;
            this.doRemove(c, this.codeSystems, CodeSystem::getUrl, CodeSystem::getVersion);
        } else if (resource instanceof StructureDefinition) {
            StructureDefinition s = (StructureDefinition)resource;
            this.doRemove(s, this.structureDefinitions, StructureDefinition::getUrl, StructureDefinition::getVersion);
        } else if (resource instanceof ValueSet) {
            ValueSet v = (ValueSet)resource;
            this.doRemove(v, this.valueSets, ValueSet::getUrl, ValueSet::getVersion);
        }
    }

    private <R extends Resource> void doRemove(R resource, ConcurrentMap<String, CacheEntry<R>> cache, Function<R, String> toUrl, Function<R, String> toVersion) {
        String url = toUrl.apply(resource);
        String version = toVersion.apply(resource);
        cache.remove(url);
        cache.remove(url + "|" + version);
    }

    private void remove(Class<? extends Resource> type, String id) {
        if (CodeSystem.class.equals(type)) {
            this.doRemove(id, this.codeSystems);
        } else if (StructureDefinition.class.equals(type)) {
            this.doRemove(id, this.structureDefinitions);
        } else if (ValueSet.class.equals(type)) {
            this.doRemove(id, this.valueSets);
        }
    }

    private <R extends Resource> void doRemove(String id, ConcurrentMap<String, CacheEntry<R>> cache) {
        String urlAndVersion = (String)this.urlAndVersionsById.get(id);
        if (urlAndVersion != null) {
            String[] split = urlAndVersion.split("\\|");
            String url = split.length > 0 ? split[0] : "";
            String version = split.length > 1 ? split[1] : "";
            cache.remove(url);
            cache.remove(url + "|" + version);
        }
    }

    public List<IBaseResource> fetchAllConformanceResources() {
        if (!this.fetchAllConformanceResourcesDone.get()) {
            logger.trace("Fetching all conformance resources");
            List allConformanceResources = this.delegate.fetchAllConformanceResources();
            allConformanceResources.stream().filter(r -> r instanceof Resource).map(r -> (Resource)r).forEach(this::update);
            this.fetchAllConformanceResourcesDone.set(true);
            this.fetchAllStructureDefinitionsDone.set(true);
            return allConformanceResources;
        }
        logger.trace("Fetching all conformance resources from cache");
        return Stream.concat(this.codeSystems.values().stream(), Stream.concat(this.structureDefinitions.values().stream(), this.valueSets.values().stream())).map(c -> c.get()).collect(Collectors.toList());
    }

    public <T extends IBaseResource> List<T> fetchAllStructureDefinitions() {
        if (!this.fetchAllStructureDefinitionsDone.get()) {
            logger.trace("Fetching all structure-definitions");
            List allStructureDefinitions = this.delegate.fetchAllStructureDefinitions();
            allStructureDefinitions.stream().filter(r -> r instanceof Resource).map(r -> (Resource)r).forEach(this::update);
            this.fetchAllStructureDefinitionsDone.set(true);
            return allStructureDefinitions;
        }
        logger.trace("Fetching all structure-definitions from cache");
        List all = this.structureDefinitions.values().stream().map(c -> c.get()).collect(Collectors.toList());
        return all;
    }

    public IBaseResource fetchStructureDefinition(String url) {
        logger.trace("Fetiching structure-definition '{}'", (Object)url);
        if (url == null || url.isBlank()) {
            return null;
        }
        return this.fetch(this.structureDefinitions, url, () -> (StructureDefinition)this.delegate.fetchStructureDefinition(url));
    }

    public boolean isCodeSystemSupported(ValidationSupportContext theRootValidationSupport, String url) {
        return this.fetchCodeSystem(url) != null;
    }

    public IBaseResource fetchCodeSystem(String url) {
        logger.trace("Fetiching code-system '{}'", (Object)url);
        if (url == null || url.isBlank()) {
            return null;
        }
        return this.fetch(this.codeSystems, url, () -> (CodeSystem)this.delegate.fetchCodeSystem(url));
    }

    public boolean isValueSetSupported(ValidationSupportContext theRootValidationSupport, String url) {
        return this.fetchValueSet(url) != null;
    }

    public IBaseResource fetchValueSet(String url) {
        logger.trace("Fetiching value-set '{}'", (Object)url);
        if (url == null || url.isBlank()) {
            return null;
        }
        return this.fetch(this.valueSets, url, () -> (ValueSet)this.delegate.fetchValueSet(url));
    }

    private <R extends Resource> R fetch(ConcurrentMap<String, CacheEntry<R>> cache, String url, Supplier<R> fetch) {
        CacheEntry cacheEntry = (CacheEntry)cache.get(url);
        if (cacheEntry != null) {
            return cacheEntry.get();
        }
        Resource resource = (Resource)fetch.get();
        if (resource == null) {
            return null;
        }
        cache.put(url, new CacheEntry<Resource>(resource, fetch));
        return (R)resource;
    }

    public IValidationSupport.ValueSetExpansionOutcome expandValueSet(ValidationSupportContext theRootValidationSupport, ValueSetExpansionOptions theExpansionOptions, IBaseResource theValueSetToExpand) {
        return this.delegate.expandValueSet(theRootValidationSupport, theExpansionOptions, theValueSetToExpand);
    }

    public <T extends IBaseResource> T fetchResource(Class<T> theClass, String theUri) {
        return (T)this.delegate.fetchResource(theClass, theUri);
    }

    public IValidationSupport.CodeValidationResult validateCode(ValidationSupportContext theRootValidationSupport, ConceptValidationOptions theOptions, String theCodeSystem, String theCode, String theDisplay, String theValueSetUrl) {
        return this.delegate.validateCode(theRootValidationSupport, theOptions, theCodeSystem, theCode, theDisplay, theValueSetUrl);
    }

    public IValidationSupport.CodeValidationResult validateCodeInValueSet(ValidationSupportContext theRootValidationSupport, ConceptValidationOptions theOptions, String theCodeSystem, String theCode, String theDisplay, IBaseResource theValueSet) {
        return this.delegate.validateCodeInValueSet(theRootValidationSupport, theOptions, theCodeSystem, theCode, theDisplay, theValueSet);
    }

    public IValidationSupport.LookupCodeResult lookupCode(ValidationSupportContext theRootValidationSupport, String theSystem, String theCode) {
        return this.delegate.lookupCode(theRootValidationSupport, theSystem, theCode);
    }

    public IBaseResource generateSnapshot(ValidationSupportContext theRootValidationSupport, IBaseResource theInput, String theUrl, String theWebUrl, String theProfileName) {
        return this.delegate.generateSnapshot(theRootValidationSupport, theInput, theUrl, theWebUrl, theProfileName);
    }

    public void invalidateCaches() {
        this.codeSystems.clear();
        this.structureDefinitions.clear();
        this.valueSets.clear();
        this.fetchAllStructureDefinitionsDone.set(false);
        this.fetchAllConformanceResourcesDone.set(false);
        this.delegate.invalidateCaches();
    }

    private static final class CacheEntry<R extends Resource> {
        final Supplier<R> resourceSupplier;
        SoftReference<R> ref;

        CacheEntry(R resource, Supplier<R> resourceSupplier) {
            this.ref = new SoftReference<R>(resource);
            this.resourceSupplier = resourceSupplier;
        }

        private SoftReference<R> read() {
            return new SoftReference<Resource>((Resource)this.resourceSupplier.get());
        }

        public R get() {
            if (this.ref == null || this.ref.get() == null) {
                this.ref = this.read();
            }
            return (R)((Resource)this.ref.get());
        }
    }
}

