package dev.dsf.bpe.webservice;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.camunda.bpm.engine.RepositoryService;
import org.camunda.bpm.engine.RuntimeService;
import org.camunda.bpm.engine.repository.ProcessDefinition;
import org.camunda.bpm.engine.runtime.ActivityInstance;
import org.camunda.bpm.engine.runtime.ProcessInstance;
import org.springframework.beans.factory.InitializingBean;
import org.thymeleaf.context.Context;

import dev.dsf.bpe.ui.ThymeleafTemplateService;
import jakarta.annotation.security.RolesAllowed;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.StreamingOutput;

@RolesAllowed("ADMIN")
@Path(RootService.PATH)
public class RootService extends AbstractService implements InitializingBean
{
	public static final String PATH = "";

	private final RepositoryService repositoryService;
	private final RuntimeService runtimeService;

	public RootService(ThymeleafTemplateService templateService, RepositoryService repositoryService,
			RuntimeService runtimeService)
	{
		super(templateService, "root");

		this.repositoryService = repositoryService;
		this.runtimeService = runtimeService;
	}

	@Override
	public void afterPropertiesSet() throws Exception
	{
		super.afterPropertiesSet();

		Objects.requireNonNull(repositoryService, "repositoryService");
		Objects.requireNonNull(runtimeService, "runtimeService");
	}

	@GET
	@Produces({ MediaType.TEXT_HTML })
	public Response root()
	{
		StreamingOutput output = write("DSF: BPE", "BPE", this::setContextValues);

		return Response.ok(output).build();
	}

	private void setContextValues(Context context)
	{
		context.setVariable("processes", processes());
		context.setVariable("processInstances", processInstances());
	}

	private record ProcessEntry(String href, String value)
	{
	}

	private List<ProcessEntry> processes()
	{
		return repositoryService.createProcessDefinitionQuery().active().unlimitedList().stream()
				.map(def -> new ProcessEntry("Process/" + def.getKey() + "/" + def.getVersionTag(),
						def.getKey() + " | " + def.getVersionTag()))
				.sorted(Comparator.comparing(ProcessEntry::value)).distinct().toList();
	}

	private List<String> processInstances()
	{
		return repositoryService.createProcessDefinitionQuery().active().unlimitedList().stream()
				.sorted(Comparator.comparing(ProcessDefinition::getKey).thenComparing(ProcessDefinition::getVersionTag))
				.flatMap(def -> runtimeService.createProcessInstanceQuery().deploymentId(def.getDeploymentId())
						.unlimitedList().stream().sorted(Comparator.comparing(ProcessInstance::getBusinessKey)).map(p ->
						{
							ActivityInstance activity = runtimeService.getActivityInstance(p.getProcessInstanceId());

							if (activity != null)
							{
								String childActivities = Stream.of(activity.getChildActivityInstances())
										.map(a -> a.getActivityType() + ":"
												+ (a.getActivityName() != null ? a.getActivityName()
														: a.getActivityId()))
										.collect(Collectors.joining(", ", "[", "]"));

								if ("[]".equals(childActivities))
									return def.getKey() + " | " + def.getVersionTag() + ": " + p.getBusinessKey()
											+ " -> " + activity.getActivityType() + ":"
											+ (activity.getActivityName() != null ? activity.getActivityName()
													: activity.getActivityId());
								else
									return def.getKey() + " | " + def.getVersionTag() + ": " + p.getBusinessKey()
											+ " -> " + childActivities;
							}
							else
								return def.getKey() + " | " + def.getVersionTag() + ": " + p.getBusinessKey();
						}))
				.toList();
	}
}
