package dev.codeflush.commons;

import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collector;
import java.util.stream.Collectors;

public final class Strings {

    private Strings() {
        throw new AssertionError("No dev.codeflush.commons.Strings instances for you!");
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and returns all parts as an array of {@link java.lang.String}
     * @see #split(String, char, Collector)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @return An array of {@link java.lang.String} containing all parts
     */
    public static String[] split(String str, char delimiter) {
        return split(str, delimiter, Collectors.toList()).toArray(new String[0]);
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and returns all parts as an array of {@link java.lang.String}
     * @see #split(String, String, Collector)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @return An array of {@link java.lang.String} containing all parts
     */
    public static String[] split(String str, String delimiter) {
        return split(str, delimiter, Collectors.toList()).toArray(new String[0]);
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and uses the given {@code collector} to create a value of type {@code R}
     * @see #split(String, char, Predicate, Collector)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @param collector The collector to create the return value
     * @param <A> The Accumulator-Type
     * @param <R> The Resulting-Type
     * @return The resulting Object created by the accumulator using the given {@code collector}
     */
    public static <A, R> R split(String str, char delimiter, Collector<String, A, R> collector) {
        return split(str, delimiter, (part) -> true, collector);
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and uses the given {@code collector} to create a value of type {@code R}
     * @see #split(String, char, ToBooleanFunction)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @param continueCondition A {@link Predicate} to test whether the splitting should continue or not
     * @param collector The collector to create the return value
     * @param <A> The Accumulator-Type
     * @param <R> The Resulting-Type
     * @return The resulting Object created by the accumulator using the given {@code collector}
     */
    public static <A, R> R split(String str, char delimiter, Predicate<String> continueCondition, Collector<String, A, R> collector) {
        final BiConsumer<A, String> accumulator = collector.accumulator();
        final A container = collector.supplier().get();

        split(str, delimiter, (part) -> {
            if (continueCondition.test(part)) {
                accumulator.accept(container, part);
                return true;
            } else {
                return false;
            }
        });

        return collector.finisher().apply(container);
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and uses the given {@code collector} to create a value of type {@code R}
     * @see #split(String, String, Predicate, Collector)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @param collector The collector to create the return value
     * @param <A> The Accumulator-Type
     * @param <R> The Resulting-Type
     * @return The resulting Object created by the accumulator using the given {@code collector}
     */
    public static <A, R> R split(String str, String delimiter, Collector<String, A, R> collector) {
        return split(str, delimiter, (part) -> true, collector);
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and uses the given {@code collector} to create a value of type {@code R}
     * @see #split(String, String, ToBooleanFunction)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @param continueCondition A {@link Predicate} to test whether the splitting should continue or not
     * @param collector The collector to create the return value
     * @param <A> The Accumulator-Type
     * @param <R> The Resulting-Type
     * @return The resulting Object created by the accumulator using the given {@code collector}
     */
    public static <A, R> R split(String str, String delimiter, Predicate<String> continueCondition, Collector<String, A, R> collector) {
        final BiConsumer<A, String> accumulator = collector.accumulator();
        final A container = collector.supplier().get();

        split(str, delimiter, (part) -> {
            if (continueCondition.test(part)) {
                accumulator.accept(container, part);
                return true;
            } else {
                return false;
            }
        });

        return collector.finisher().apply(container);
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and passes all parts to the given {@code consumer}
     * @see #split(String, char, ToBooleanFunction)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @param consumer The consumer receiving all parts
     */
    public static void split(String str, char delimiter, Consumer<String> consumer) {
        split(str, delimiter, (part) -> {
            consumer.accept(part);
            return true;
        });
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and passes all parts to the given {@code consumer},
     * as long as the {@code consumer} returns {@code true}
     * @see #split(String, String, ToBooleanFunction)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @param consumer The consumer receiving all parts (until in returns {@code false})
     */
    public static void split(String str, char delimiter, ToBooleanFunction<String> consumer) {
        int offset = 0;
        int position;
        boolean shouldContinue = true;

        while (shouldContinue && (position = str.indexOf(delimiter, offset)) != -1) {
            shouldContinue = consumer.apply(str.substring(offset, position));
            offset = position + 1;
        }

        if (shouldContinue) {
            consumer.apply(str.substring(offset));
        }
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and passes all parts to the given {@code consumer}
     * @see #split(String, String, ToBooleanFunction)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @param consumer The consumer receiving all parts
     */
    public static void split(String str, String delimiter, Consumer<String> consumer) {
        split(str, delimiter, (part) -> {
            consumer.accept(part);
            return true;
        });
    }

    /**
     * Split the given {@code str} using the {@code delimiter} and passes all parts to the given {@code consumer},
     * as long as the {@code consumer} returns {@code true}
     * @see #split(String, char, ToBooleanFunction)
     * @param str The string to split
     * @param delimiter The delimiter to split at
     * @param consumer The consumer receiving all parts (until in returns {@code false})
     */
    public static void split(String str, String delimiter, ToBooleanFunction<String> consumer) {
        final int delimiterLength = delimiter.length();
        int offset = 0;
        int position;
        boolean shouldContinue = true;

        while (shouldContinue && (position = str.indexOf(delimiter, offset)) != -1) {
            shouldContinue = consumer.apply(str.substring(offset, position));
            offset = position + delimiterLength;
        }

        if (shouldContinue) {
            consumer.apply(str.substring(offset));
        }
    }

    @FunctionalInterface
    public interface ToBooleanFunction<T> {

        boolean apply(T value);
    }
}
