package net.binis.codegen.enrich.handler;

/*-
 * #%L
 * code-generator
 * %%
 * Copyright (C) 2021 Binis Belev
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.github.javaparser.ast.body.AnnotationDeclaration;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.expr.*;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.utils.StringEscapeUtils;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import net.binis.codegen.annotation.validation.*;
import net.binis.codegen.enrich.ValidationEnricher;
import net.binis.codegen.enrich.handler.base.BaseEnricher;
import net.binis.codegen.exception.GenericCodeGenException;
import net.binis.codegen.generation.core.Helpers;
import net.binis.codegen.generation.core.interfaces.PrototypeDescription;
import net.binis.codegen.generation.core.interfaces.PrototypeField;
import net.binis.codegen.test.TestClassLoader;
import net.binis.codegen.tools.Holder;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Triple;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static net.binis.codegen.generation.core.Helpers.getExternalClassName;
import static net.binis.codegen.tools.Reflection.loadClass;
import static net.binis.codegen.tools.Tools.*;

@Slf4j
public class ValidationEnricherHandler extends BaseEnricher implements ValidationEnricher {

    private static final String VALUE = "value";
    private static final String PARAMS = "params";
    private static final String MESSAGE = "message";
    private static final String MESSAGES = "messages";
    private static final String AS_CODE = "asCode";

    private static final TestClassLoader loader = new TestClassLoader();

    @Override
    public void enrich(PrototypeDescription<ClassOrInterfaceDeclaration> description) {
        //Do nothing
    }

    @Override
    public void finalizeEnrich(PrototypeDescription<ClassOrInterfaceDeclaration> description) {
        description.getFields().forEach(f -> handleField(description, f));
    }

    @Override
    public int order() {
        return 0;
    }

    private void handleField(PrototypeDescription<ClassOrInterfaceDeclaration> description, PrototypeField field) {
        field.getDescription().getAnnotations().stream().filter(this::isValidationAnnotation).forEach(a -> processAnnotation(description, field, a));
    }

    private void processAnnotation(PrototypeDescription<ClassOrInterfaceDeclaration> description, PrototypeField field, AnnotationExpr annotation) {
        var name = Helpers.getExternalClassNameIfExists(annotation.findCompilationUnit().get(), annotation.getNameAsString());

        var cls = loadClass(name);
        if (nonNull(cls)) {
            if (Validate.class.equals(cls) || cls.isAnnotationPresent(Validate.class)) {
                generateValidation(field, annotation, cls);
            } else if (Sanitize.class.equals(cls) || cls.isAnnotationPresent(Sanitize.class)) {
                generateSanitization(field, annotation, cls);
            } else if (Execute.class.equals(cls) || cls.isAnnotationPresent(Execute.class)) {
                generateExecution(field, annotation, cls);
            }
        } else {
            notNull(lookup.findExternal(name), d ->
                    handleAnnotationFromSource(d.getDeclaration().asAnnotationDeclaration(), field, annotation));
        }
    }

    private void handleAnnotationFromSource(AnnotationDeclaration decl, PrototypeField field, AnnotationExpr annotation) {
        var ann = decl.getAnnotationByClass(Validate.class);
        if (ann.isPresent()) {
            generateValidation(field, annotation, ann.get(), decl);
        } else {
            ann = decl.getAnnotationByClass(Sanitize.class);
            if (ann.isPresent()) {
                generateSanitization(field, annotation, ann.get(), decl);
            } else {
                ann = decl.getAnnotationByClass(Execute.class);
                if (ann.isPresent()) {
                    generateExecution(field, annotation, ann.get(), decl);
                }
            }
        }
    }

    private void generateSanitization(PrototypeField field, AnnotationExpr annotation, AnnotationExpr ann, AnnotationDeclaration annotationClass) {
        var params = getSanitizationParams(field, annotation, ann, annotationClass);
        field.getDeclaration().findCompilationUnit().ifPresent(u -> u.addImport(getExternalClassName(annotationClass.findCompilationUnit().get(), params.getCls())));
        generateSanitization(field, params);
    }

    private void generateSanitization(PrototypeField field, AnnotationExpr annotation, Class<?> annotationClass) {
        generateSanitization(field, getSanitizationParams(field, annotation, annotationClass));
    }

    private void generateSanitization(PrototypeField field, Params params) {
        if (isNull(field.getImplementationSetter())) {
            field.generateSetter();
        }

        addSanitization(field, field.getImplementationSetter(), params, ModifierType.MAIN);
        field.getModifiers().forEach(modifier -> addSanitization(field, modifier, params, ModifierType.MODIFIER));
    }

    private Params getSanitizationParams(PrototypeField field, AnnotationExpr annotation, AnnotationExpr ann, AnnotationDeclaration annotationClass) {
        var params = Params.builder();

        handleSanitizationAnnotation(ann, params);
        //TODO: Handle aliases

        return params.build();
    }

    private Params getSanitizationParams(PrototypeField field, AnnotationExpr annotation, Class<?> annotationClass) {
        String cls = null;
        var params = Params.builder();

        if (!Sanitize.class.equals(annotationClass)) {
            var ann = annotationClass.getDeclaredAnnotation(Sanitize.class);
            params.cls(ann.value().getSimpleName()).params(Arrays.asList(ann.params()));
            cls = ann.value().getCanonicalName();
            field.getDeclaration().findCompilationUnit().get().addImport(cls);

            handleAliases(annotation, annotationClass, params);
        } else {
            handleSanitizationAnnotation(annotation, params);
        }

        var result = params.build();

        if (isNull(result.getAsCode())) {
            notNull(loadClass(isNull(cls) ? getExternalClassName(field.getParsed().getDeclaration().findCompilationUnit().get(), result.getCls()) : cls), c ->
                    notNull(c.getDeclaredAnnotation(AsCode.class), a -> result.setAsCode(a.value())));
        }

        return result;
    }

    private void handleSanitizationAnnotation(AnnotationExpr annotation, Params.ParamsBuilder params) {
        for (var node : annotation.getChildNodes()) {
            if (node instanceof ClassExpr) {
                params.cls(((ClassExpr) node).getTypeAsString());
            } else if (node instanceof MemberValuePair) {
                var pair = (MemberValuePair) node;
                switch (pair.getNameAsString()) {
                    case VALUE:
                        params.cls(pair.getValue().asClassExpr().getTypeAsString());
                        break;
                    case PARAMS:
                        params.params(pair.getValue().asArrayInitializerExpr().getValues().stream().map(Expression::asStringLiteralExpr).map(StringLiteralExpr::asString).collect(Collectors.toList()));
                        break;
                    case AS_CODE:
                        params.asCode(pair.getValue().asStringLiteralExpr().asString());
                        break;
                    default:
                }
            }
        }
    }

    private Params getValidationParams(PrototypeField field, AnnotationExpr annotation, AnnotationExpr ann, AnnotationDeclaration annotationClass) {
        var params = Params.builder();

        handleValidationAnnotation(ann, params);
        //TODO: Handle aliases

        return params.build();
    }

    private Params getValidationParams(PrototypeField field, AnnotationExpr annotation, Class<?> annotationClass) {
        var params = Params.builder();
        String cls = null;

        if (!Validate.class.equals(annotationClass)) {
            var ann = annotationClass.getDeclaredAnnotation(Validate.class);
            params.cls(ann.value().getSimpleName()).params(Arrays.asList(ann.params())).message(ann.message());
            cls = ann.value().getCanonicalName();
            field.getDeclaration().findCompilationUnit().get().addImport(cls);

            handleAliases(annotation, annotationClass, params);
        } else {
            handleValidationAnnotation(annotation, params);
        }

        var result = params.build();

        if (isNull(result.getAsCode())) {
            notNull(loadClass(isNull(cls) ? getExternalClassName(field.getParsed().getDeclaration().findCompilationUnit().get(), result.getCls()) : cls), c ->
                    notNull(c.getDeclaredAnnotation(AsCode.class), a -> result.setAsCode(a.value())));
        }

        return result;
    }

    private void handleValidationAnnotation(AnnotationExpr annotation, ValidationEnricherHandler.Params.ParamsBuilder params) {
        for (var node : annotation.getChildNodes()) {
            if (node instanceof ClassExpr) {
                params.cls(((ClassExpr) node).getTypeAsString());
            } else if (node instanceof MemberValuePair) {
                var pair = (MemberValuePair) node;
                switch (pair.getNameAsString()) {
                    case VALUE:
                        params.cls(pair.getValue().asClassExpr().getTypeAsString());
                        break;
                    case MESSAGE:
                        params.message(pair.getValue().asStringLiteralExpr().asString());
                        break;
                    case MESSAGES:
                        params.messages(pair.getValue().asArrayInitializerExpr().getValues().stream().map(e -> e.asStringLiteralExpr().asString()).collect(Collectors.toList()));
                        break;
                    case PARAMS:
                        if (pair.getValue().isArrayInitializerExpr()) {
                            params.params(pair.getValue().asArrayInitializerExpr().getValues().stream().map(Expression::asStringLiteralExpr).map(StringLiteralExpr::asString).collect(Collectors.toList()));
                        } else {
                            params.params(List.of(pair.getValue()));
                        }
                        break;
                    case AS_CODE:
                        params.asCode(pair.getValue().asStringLiteralExpr().asString());
                        break;
                    default:
                }
            }
        }
    }

    private void handleAliases(AnnotationExpr annotation, Class<?> annotationClass, Params.ParamsBuilder params) {
        var list = new ArrayList<Object>();
        var parOrder = Arrays.stream(annotationClass.getDeclaredMethods())
                .filter(m -> Arrays.stream(m.getDeclaredAnnotations()).filter(a -> a.annotationType().isAssignableFrom(AliasFor.class)).map(AliasFor.class::cast).anyMatch(a -> PARAMS.equals(a.value())))
                .map(m -> Triple.of(m.getName(), m.getDefaultValue(), m.getDeclaredAnnotation(AsCode.class)))
                .collect(Collectors.toList());

        Arrays.stream(annotationClass.getDeclaredMethods())
                .filter(m -> MESSAGE.equals(m.getName()))
                .filter(m -> m.getReturnType().equals(String.class))
                .filter(m -> isNull(m.getDeclaredAnnotation(AliasFor.class)))
                .findFirst().ifPresent(m ->
                        params.message((String) m.getDefaultValue()));

        var messages = Holder.<List<String>>blank();
        Arrays.stream(annotationClass.getDeclaredMethods())
                .filter(m -> MESSAGES.equals(m.getName()))
                .filter(m -> m.getReturnType().equals(String[].class))
                .filter(m -> isNull(m.getDeclaredAnnotation(AliasFor.class)))
                .findFirst().ifPresent(m ->
                        messages.set(List.of((String[]) m.getDefaultValue())));

        if (messages.isEmpty()) {
            Arrays.stream(annotationClass.getDeclaredMethods())
                    .filter(m -> m.getReturnType().equals(String.class))
                    .filter(m -> nullCheck(m.getDeclaredAnnotation(AliasFor.class), a -> MESSAGES.equals(a.value()), false))
                    .forEach(m -> {
                        if (messages.isEmpty()) {
                            messages.set(new ArrayList<>());
                        }
                        if (nonNull(m.getDefaultValue())) {
                            messages.get().add(m.getDefaultValue().toString());
                        } else {
                            messages.get().add("(%s) Invalid value!");
                        }
                    });
        }

        parOrder.forEach(p -> list.add(checkAsCode(p.getMiddle(), p.getRight())));
        var msgs = 0;

        for (var node : annotation.getChildNodes()) {
            if (node instanceof MemberValuePair) {
                var pair = (MemberValuePair) node;
                switch (Arrays.stream(annotationClass.getDeclaredMethods())
                        .filter(m -> m.getName().equals(pair.getNameAsString()))
                        .map(m -> m.getDeclaredAnnotation(AliasFor.class))
                        .filter(Objects::nonNull)
                        .map(AliasFor::value)
                        .findFirst()
                        .orElseGet(pair::getNameAsString)) {
                    case VALUE:
                        params.cls(pair.getValue().asClassExpr().getTypeAsString());
                        break;
                    case MESSAGE:
                        params.message(pair.getValue().asStringLiteralExpr().asString());
                        break;
                    case MESSAGES:
                        if (pair.getValue().isArrayInitializerExpr()) {
                            params.messages(pair.getValue().asArrayInitializerExpr().getValues().stream().map(e -> e.asStringLiteralExpr().asString()).collect(Collectors.toList()));
                        } else if (pair.getValue().isStringLiteralExpr()) {
                            var msg = pair.getValue().asStringLiteralExpr().asString();
                            if (messages.isEmpty()) {
                                messages.set(new ArrayList<>());
                            }
                            if (msgs < messages.get().size()) {
                                messages.get().set(msgs, msg);
                            } else {
                                messages.get().add(msg);
                            }
                            msgs++;
                        }
                        break;
                    case AS_CODE:
                        params.asCode(pair.getValue().asStringLiteralExpr().asString());
                        break;
                    case PARAMS:
                        if (pair.getValue().isArrayInitializerExpr()) {
                            list.addAll(pair.getValue().asArrayInitializerExpr().getValues().stream()
                                    .map(Expression::asStringLiteralExpr)
                                    .map(StringLiteralExpr::asString)
                                    .collect(Collectors.toList()));
                        } else {
                            var idx = getParamIndex(parOrder, pair.getNameAsString());
                            if (idx != -1) {
                                list.set(idx, getParamValue(pair.getValue()));
                            } else {
                                throw new GenericCodeGenException("Invalid annotation params! " + annotation);
                            }
                        }
                        break;
                    default:
                }
            } else if (node instanceof StringLiteralExpr) {
                var exp = ((StringLiteralExpr) node).asString();
                switch (Arrays.stream(annotationClass.getDeclaredMethods())
                        .filter(m -> m.getName().equals(VALUE))
                        .map(m -> m.getDeclaredAnnotation(AliasFor.class))
                        .filter(Objects::nonNull)
                        .map(AliasFor::value)
                        .findFirst()
                        .orElse(VALUE)) {
                    case VALUE:
                        params.cls(exp);
                        break;
                    case MESSAGE:
                        params.message(exp);
                        break;
                    case AS_CODE:
                        params.asCode(exp);
                        break;
                    case PARAMS:
                        var idx = getParamIndex(parOrder, VALUE);
                        if (idx != -1) {
                            var triple = parOrder.get(idx);
                            if (nonNull(triple.getRight())) {
                                list.set(idx, checkAsCode(exp, triple.getRight()));
                            } else {
                                list.set(idx, exp);
                            }
                        } else {
                            throw new GenericCodeGenException("Invalid annotation params! " + annotation);
                        }
                        break;
                    default:

                }
            }
        }
        if (!list.isEmpty()) {
            params.params(list);
        }
        params.messages = messages.get();
    }

    private Object checkAsCode(Object value, AsCode code) {
        if (nonNull(code)) {
            return AsCodeHolder.builder().value((String) value).format(code.value()).build();
        }
        return value;
    }

    private Object getParamValue(Expression value) {
        if (value.isStringLiteralExpr()) {
            return value.asStringLiteralExpr().asString();
        } else if (value.isIntegerLiteralExpr()) {
            return value.asIntegerLiteralExpr().asNumber();
        } else if (value.isDoubleLiteralExpr()) {
            return value.asDoubleLiteralExpr().asDouble();
        } else if (value.isBooleanLiteralExpr()) {
            return value.asBooleanLiteralExpr().getValue();
        }
        //TODO: Handle external constants
        return null;
    }

    private int getParamIndex(List<Triple<String, Object, AsCode>> list, String name) {
        for (var i = 0; i < list.size(); i++) {
            if (name.equals(list.get(i).getLeft())) {
                return i;
            }
        }
        return -1;
    }

    private void generateValidation(PrototypeField field, AnnotationExpr annotation, AnnotationExpr ann, AnnotationDeclaration annotationClass) {
        var params = getValidationParams(field, annotation, ann, annotationClass);
        field.getDeclaration().findCompilationUnit().ifPresent(u -> u.addImport(getExternalClassName(annotationClass.findCompilationUnit().get(), params.getCls())));
        generateValidation(field, params);
    }

    private void generateValidation(PrototypeField field, AnnotationExpr annotation, Class<?> annotationClass) {
        generateValidation(field, getValidationParams(field, annotation, annotationClass));
    }

    private void generateValidation(PrototypeField field, Params params) {
        if (isNull(field.getImplementationSetter())) {
            field.generateSetter();
        }

        addValidation(field, field.getImplementationSetter(), params, ModifierType.MAIN);
        field.getModifiers().forEach(modifier -> addValidation(field, modifier, params, ModifierType.MODIFIER));
    }


    private void addValidation(PrototypeField field, MethodDeclaration method, Params params, ModifierType modifier) {
        method.findCompilationUnit().get().addImport("net.binis.codegen.validation.flow.Validation");
        var block = method.getChildNodes().stream().filter(BlockStmt.class::isInstance).map(BlockStmt.class::cast).findFirst().get();

        if (block.getStatements().get(0).asExpressionStmt().getExpression() instanceof AssignExpr) {
            var exp = new StringBuilder("Validation.start(\"")
                    .append(field.getName())
                    .append("\", ")
                    .append(field.getName())
                    .append(").validate")
                    .append(nonNull(params.getMessages()) ? "WithMessages(" : "(")
                    .append(params.getCls())
                    .append(".class, ")
                    .append(calcMessage(params))
                    .append(buildParamsStr(params, field, modifier))
                    .append(").perform(v -> this.map = v);");
            var expr = lookup.getParser().parseStatement(exp.toString()).getResult().get();
            var original = block.getStatements().remove(0);
            ((ExpressionStmt) original).getExpression().asAssignExpr().setValue(new NameExpr("v"));
            var mCall = expr.asExpressionStmt().getExpression().asMethodCallExpr();
            ((LambdaExpr) mCall.getChildNodes().get(mCall.getChildNodes().size() - 1)).setBody(original);
            block.getStatements().add(0, expr);
        } else {
            var mCall = block.getStatements().get(0).asExpressionStmt().getExpression().asMethodCallExpr();
            var chain = mCall.getScope().get();
            mCall.removeScope();
            var m = new MethodCallExpr(chain, "validate" + (nonNull(params.getMessages()) ? "WithMessages" : "")).addArgument(params.getCls() + ".class").addArgument(calcMessage(params));
            notNull(params.getParams(), p -> p.forEach(param ->
                    m.addArgument(buildParamsStr(param, params, field, modifier))));
            mCall.setScope(m);
        }
    }

    private String calcMessage(Params params) {
        if (nonNull(params.getMessages())) {
            return "new String[] {" + params.messages.stream().map(s -> "\"" + StringEscapeUtils.escapeJava(s) + "\"").collect(Collectors.joining(", ")) + "}";
        } else {
            return isNull(params.getMessage()) ? "null" : "\"" + StringEscapeUtils.escapeJava(params.getMessage()) + "\"";
        }
    }

    private void addSanitization(PrototypeField field, MethodDeclaration method, Params params, ModifierType modifier) {
        method.findCompilationUnit().get().addImport("net.binis.codegen.validation.flow.Validation");
        var block = method.getChildNodes().stream().filter(BlockStmt.class::isInstance).map(BlockStmt.class::cast).findFirst().get();

        if (block.getStatements().get(0).asExpressionStmt().getExpression() instanceof AssignExpr) {
            var exp = new StringBuilder("Validation.start(\"")
                    .append(field.getName())
                    .append("\", ")
                    .append(field.getName())
                    .append(").sanitize(")
                    .append(params.getCls())
                    .append(".class")
                    .append(buildParamsStr(params, field, modifier))
                    .append(").perform(v -> this.map = v);");
            var expr = lookup.getParser().parseStatement(exp.toString()).getResult().get();
            var original = block.getStatements().remove(0);
            ((ExpressionStmt) original).getExpression().asAssignExpr().setValue(new NameExpr("v"));
            var mCall = expr.asExpressionStmt().getExpression().asMethodCallExpr();
            ((LambdaExpr) mCall.getChildNodes().get(mCall.getChildNodes().size() - 1)).setBody(original);
            block.getStatements().add(0, expr);
        } else {
            var mCall = block.getStatements().get(0).asExpressionStmt().getExpression().asMethodCallExpr();
            var chain = mCall.getScope().get();
            mCall.removeScope();
            var m = new MethodCallExpr(chain, "sanitize").addArgument(params.getCls() + ".class");
            notNull(params.getParams(), p -> p.forEach(param ->
                    m.addArgument(buildParamsStr(param, params, field, modifier))));
            mCall.setScope(m);
        }
    }

    private void generateExecution(PrototypeField field, AnnotationExpr annotation, AnnotationExpr ann, AnnotationDeclaration annotationClass) {
        var params = getExecutionParams(field, annotation, ann, annotationClass);
        field.getDeclaration().findCompilationUnit().ifPresent(u -> u.addImport(getExternalClassName(annotationClass.findCompilationUnit().get(), params.getCls())));
        generateExecution(field, params);
    }

    private void generateExecution(PrototypeField field, AnnotationExpr annotation, Class<?> annotationClass) {
        generateExecution(field, getExecutionParams(field, annotation, annotationClass));
    }

    private void generateExecution(PrototypeField field, Params params) {
        if (isNull(field.getImplementationSetter())) {
            field.generateSetter();
        }

        addExecution(field, field.getImplementationSetter(), params, ModifierType.MAIN);
        field.getModifiers().forEach(modifier -> addExecution(field, modifier, params, ModifierType.MODIFIER));
    }

    private void addExecution(PrototypeField field, MethodDeclaration method, Params params, ModifierType modifier) {
        method.findCompilationUnit().get().addImport("net.binis.codegen.validation.flow.Validation");
        var block = method.getChildNodes().stream().filter(BlockStmt.class::isInstance).map(BlockStmt.class::cast).findFirst().get();

        if (block.getStatements().get(0).asExpressionStmt().getExpression() instanceof AssignExpr) {
            var exp = new StringBuilder("Validation.start(\"")
                    .append(field.getName())
                    .append("\", ")
                    .append(field.getName())
                    .append(").execute(")
                    .append(params.getCls())
                    .append(".class, ")
                    .append(calcMessage(params))
                    .append(buildParamsStr(params, field, modifier))
                    .append(").perform(v -> this.map = v);");
            var expr = lookup.getParser().parseStatement(exp.toString()).getResult().get();
            var original = block.getStatements().remove(0);
            ((ExpressionStmt) original).getExpression().asAssignExpr().setValue(new NameExpr("v"));
            var mCall = expr.asExpressionStmt().getExpression().asMethodCallExpr();
            ((LambdaExpr) mCall.getChildNodes().get(mCall.getChildNodes().size() - 1)).setBody(original);
            block.getStatements().add(0, expr);
        } else {
            var mCall = block.getStatements().get(0).asExpressionStmt().getExpression().asMethodCallExpr();
            var chain = mCall.getScope().get();
            mCall.removeScope();
            var m = new MethodCallExpr(chain, "execute").addArgument(params.getCls() + ".class").addArgument(calcMessage(params));
            notNull(params.getParams(), p -> p.forEach(param ->
                    m.addArgument(buildParamsStr(param, params, field, modifier))));
            mCall.setScope(m);
        }
    }

    private Params getExecutionParams(PrototypeField field, AnnotationExpr annotation, AnnotationExpr ann, AnnotationDeclaration annotationClass) {
        var params = Params.builder();

        handleExecutionAnnotation(ann, params);
        //TODO: Handle aliases

        return params.build();
    }

    private Params getExecutionParams(PrototypeField field, AnnotationExpr annotation, Class<?> annotationClass) {
        var params = Params.builder();
        String cls = null;

        if (!Execute.class.equals(annotationClass)) {
            var ann = annotationClass.getDeclaredAnnotation(Execute.class);
            params.cls(ann.value().getSimpleName()).params(Arrays.asList(ann.params()));
            cls = ann.value().getCanonicalName();
            field.getDeclaration().findCompilationUnit().get().addImport(cls);

            handleAliases(annotation, annotationClass, params);
        } else {
            handleExecutionAnnotation(annotation, params);
        }

        var result = params.build();

        if (isNull(result.getAsCode())) {
            notNull(loadClass(isNull(cls) ? getExternalClassName(field.getParsed().getDeclaration().findCompilationUnit().get(), result.getCls()) : cls), c ->
                    notNull(c.getDeclaredAnnotation(AsCode.class), a -> result.setAsCode(a.value())));
        }

        return result;
    }

    private void handleExecutionAnnotation(AnnotationExpr annotation, Params.ParamsBuilder params) {
        for (var node : annotation.getChildNodes()) {
            if (node instanceof ClassExpr) {
                params.cls(((ClassExpr) node).getTypeAsString());
            } else if (node instanceof MemberValuePair) {
                var pair = (MemberValuePair) node;
                switch (pair.getNameAsString()) {
                    case VALUE:
                        params.cls(pair.getValue().asClassExpr().getTypeAsString());
                        break;
                    case MESSAGE:
                        params.message(pair.getValue().asStringLiteralExpr().asString());
                        break;
                    case PARAMS:
                        params.params(pair.getValue().asArrayInitializerExpr().getValues().stream().map(Expression::asStringLiteralExpr).map(StringLiteralExpr::asString).collect(Collectors.toList()));
                        break;
                    case AS_CODE:
                        params.asCode(pair.getValue().asStringLiteralExpr().asString());
                        break;
                    default:
                }
            }
        }
    }

    private String buildParamsStr(Params params, PrototypeField field, ModifierType modifier) {
        var list = params.getParams();
        if (isNull(list) || list.isEmpty()) {
            return "";
        }

        var result = new StringBuilder();
        for (var param : list) {
            if (param instanceof String) {
                result.append(", \"")
                        .append(StringEscapeUtils.escapeJava((String) param))
                        .append("\"");
            } else if (param instanceof AsCodeHolder) {
                var holder = (AsCodeHolder) param;
                var format = "%s".equals(holder.getFormat()) && !StringUtils.isBlank(params.getAsCode()) ? params.getAsCode() : holder.getFormat();
                result.append(", ")
                        .append(String.format(format.replaceAll("\\{type}", field.getDeclaration().getVariable(0).getTypeAsString()),
                                holder.getValue()
                                        .replaceAll("\\{type}", field.getDeclaration().getVariable(0).getTypeAsString())
                                        .replaceAll("\\{entity}", (ModifierType.MODIFIER.equals(modifier) ? "(" + field.getDeclaration().findAncestor(ClassOrInterfaceDeclaration.class).get().getNameAsString() + ")" : "") + modifier.getValue())));
            } else {
                result.append(", ")
                        .append(nonNull(param) ? param.toString() : "null");
            }
        }
        return result.toString();
    }

    private String buildParamsStr(Object param, Params params, PrototypeField field, ModifierType modifier) {
        if (param instanceof String) {
            return "\"" + StringEscapeUtils.escapeJava((String) param) + "\"";
        } else if (param instanceof AsCodeHolder) {
            var holder = (AsCodeHolder) param;
            var format = "%s".equals(holder.getFormat()) && !StringUtils.isBlank(params.getAsCode()) ? params.getAsCode() : holder.getFormat();
            return String.format(format.replaceAll("\\{type}", field.getDeclaration().getVariable(0).getTypeAsString()), holder.getValue());
        } else {
            return nonNull(param) ? param.toString() : "null";
        }
    }


    private boolean isValidationAnnotation(AnnotationExpr annotation) {
        var name = Helpers.getExternalClassNameIfExists(annotation.findCompilationUnit().get(), annotation.getNameAsString());
        var external = lookup.findExternal(name);
        if (nonNull(external)) {
            return withRes(external.getDeclaration(), decl ->
                    decl.isAnnotationPresent(Validate.class) || decl.isAnnotationPresent(Sanitize.class) || decl.isAnnotationPresent(Execute.class));
        }
        return withRes(loadClass(name), cls ->
                Validate.class.equals(cls) || cls.isAnnotationPresent(Validate.class) || Sanitize.class.equals(cls) || cls.isAnnotationPresent(Sanitize.class) || Execute.class.equals(cls) || cls.isAnnotationPresent(Execute.class), false);
    }

    private static enum ModifierType {
        MAIN("this"),
        MODIFIER("parent");

        private String value;

        ModifierType(String s) {
            this.value = s;
        }

        String getValue() {
            return value;
        }
    }

    @Data
    @Builder
    private static class Params {
        private String cls;
        private String message;
        private List<String> messages;

        private List<Object> params;
        private String asCode;

        //Custom builder to satisfy java-doc
        public static class ParamsBuilder {

        }
    }

    @Data
    @Builder
    private static class AsCodeHolder {
        private String value;
        private String format;
    }

}
