package net.binis.codegen.hibernate;

/*-
 * #%L
 * code-generator-hibernate
 * %%
 * Copyright (C) 2021 - 2022 Binis Belev
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import net.binis.codegen.objects.base.enumeration.CodeEnum;
import org.hibernate.engine.spi.SharedSessionContractImplementor;
import org.hibernate.type.descriptor.ValueBinder;
import org.hibernate.type.descriptor.ValueExtractor;
import org.hibernate.type.descriptor.sql.IntegerTypeDescriptor;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

public class OrdinalCodeEnumValueConverter<E extends CodeEnum> implements CodeEnumValueConverter<E,Integer>, Serializable {

	private final CodeEnumJavaTypeDescriptor<E> enumJavaDescriptor;

	private transient ValueExtractor<E> valueExtractor;

	private transient ValueBinder<Integer> valueBinder;

	public OrdinalCodeEnumValueConverter(CodeEnumJavaTypeDescriptor<E> enumJavaDescriptor) {
		this.enumJavaDescriptor = enumJavaDescriptor;
		this.valueExtractor = createValueExtractor( enumJavaDescriptor );
		this.valueBinder = createValueBinder();
	}

	@Override
	public E toDomainValue(Integer relationalForm) {
		return enumJavaDescriptor.fromOrdinal( relationalForm );
	}

	@Override
	public Integer toRelationalValue(E domainForm) {
		return enumJavaDescriptor.toOrdinal( domainForm );
	}

	@Override
	public int getJdbcTypeCode() {
		return Types.INTEGER;
	}

	@Override
	public CodeEnumJavaTypeDescriptor<E> getJavaDescriptor() {
		return enumJavaDescriptor;
	}

	@Override
	public E readValue(ResultSet resultSet, String name, SharedSessionContractImplementor session) throws SQLException {
		return valueExtractor.extract( resultSet, name, session );
	}

	@Override
	public void writeValue(PreparedStatement statement, E value, int position, SharedSessionContractImplementor session) throws SQLException {
		final Integer jdbcValue = value == null ? null : toRelationalValue( value );

		valueBinder.bind( statement, jdbcValue, position, session );
	}

	@Override
	@SuppressWarnings("unchecked")
	public String toSqlLiteral(Object value) {
		return Integer.toString( ( (E) value ).ordinal() );
	}

	private static <T extends CodeEnum> ValueExtractor<T> createValueExtractor(CodeEnumJavaTypeDescriptor<T> enumJavaDescriptor) {
		return IntegerTypeDescriptor.INSTANCE.getExtractor( enumJavaDescriptor );
	}

	private static ValueBinder<Integer> createValueBinder() {
		return IntegerTypeDescriptor.INSTANCE.getBinder( org.hibernate.type.descriptor.java.IntegerTypeDescriptor.INSTANCE );
	}

	private void readObject(ObjectInputStream stream) throws ClassNotFoundException, IOException {
		stream.defaultReadObject();

		this.valueExtractor = createValueExtractor( enumJavaDescriptor );
		this.valueBinder = createValueBinder();
	}
}
