package dev.array21.pluginstatlib;

import java.io.IOException;
import java.util.function.Consumer;

import com.google.gson.Gson;

import nl.thedutchmc.httplib.Http;
import nl.thedutchmc.httplib.Http.MediaFormat;
import nl.thedutchmc.httplib.Http.RequestMethod;
import nl.thedutchmc.httplib.Http.ResponseObject;

public class PluginStat {

	private String uuid = null;
	private Consumer<String> setUuidFn = null;
	private Consumer<Object> logErrFn = null;
	
	private static final String SERVER = "https://plugin-stat.k8s.array21.dev";
	private static final int INTERVAL_MINUTES = 2;
	
	private static volatile boolean STOPPING = false;

	
	public void start() {
		assert this.uuid != null;
		assert this.setUuidFn != null;
		assert this.logErrFn != null;
		
		Runnable r = new Runnable() {
			private final Http HTTP = new Http();
			private final Gson GSON = new Gson();
			
			@Override
			public void run() {
				while(!PluginStat.STOPPING) {
					StatPackage statPackage = new StatPackage(PluginStat.this.uuid);
					String statSerialized = GSON.toJson(statPackage);
					
					ResponseObject response;
					try {
						response = HTTP.makeRequest(RequestMethod.POST, PluginStat.SERVER, null, MediaFormat.JSON, statSerialized, null);
					} catch(IOException e) {
						PluginStat.this.logErrFn.accept("Failed to send metrics.");
						sleep();
						continue;
					}
					
					if(response.getResponseCode() != 200) {
						codeSwitch: switch(response.getResponseCode()) {
						case 409:
							InvalidUuidResponse r = GSON.fromJson(response.getMessage(), InvalidUuidResponse.class);
							PluginStat.this.uuid = r.getNewUuid();
							PluginStat.this.setUuidFn.accept(r.getNewUuid());
							
							continue;
						default: 
							break codeSwitch;
						}
					}
					
					sleep();
				}
			}
			
			private void sleep() {
				try {
					Thread.sleep(INTERVAL_MINUTES * 60 * 1000);
				} catch(InterruptedException e) {}
			}
		};
		
		Thread t = new Thread(r, "PluginStatThread");
		t.start();
	}
	
	public static void disable() {
		PluginStat.STOPPING = true;
	}
	

	public static class PluginStatBuilder {
		private PluginStat pluginStat = new PluginStat();
		
		public static PluginStatBuilder createDefault() {
			return new PluginStatBuilder();
		}
		
		public PluginStatBuilder setUuid(String uuid) {
			this.pluginStat.uuid = uuid;
			return this;
		}
		
		public PluginStatBuilder setSetUuidFn(Consumer<String> fn) {
			this.pluginStat.setUuidFn = fn;
			return this;
		}
		
		public PluginStatBuilder setLogErrFn(Consumer<Object> fn) {
			this.pluginStat.logErrFn = fn;
			return this;
		}
		
		public PluginStat builder() {
			assert this.pluginStat.uuid != null;
			assert this.pluginStat.setUuidFn != null;
			assert this.pluginStat.logErrFn != null;
			
			return this.pluginStat;
		}
	}
}
