/*
 * Decompiled with CFR 0.152.
 */
package org.to2mbn.jmccc.mcdownloader.download.cache;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.ref.SoftReference;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.to2mbn.jmccc.mcdownloader.download.Downloader;
import org.to2mbn.jmccc.mcdownloader.download.cache.provider.CacheProvider;
import org.to2mbn.jmccc.mcdownloader.download.concurrent.CompletedFuture;
import org.to2mbn.jmccc.mcdownloader.download.concurrent.DownloadCallback;
import org.to2mbn.jmccc.mcdownloader.download.tasks.DownloadSession;
import org.to2mbn.jmccc.mcdownloader.download.tasks.DownloadTask;

class CachedDownloader
implements Downloader {
    private static final Logger LOGGER = Logger.getLogger(CachedDownloader.class.getCanonicalName());
    private Downloader upstream;
    private CacheProvider<URI, byte[]> cacheProvider;

    public CachedDownloader(Downloader upstream, CacheProvider<URI, byte[]> cacheProvider) {
        this.upstream = Objects.requireNonNull(upstream);
        this.cacheProvider = Objects.requireNonNull(cacheProvider);
    }

    @Override
    public <T> Future<T> download(DownloadTask<T> task, DownloadCallback<T> callback) {
        return this.downloadIfNecessary(task, callback, -1);
    }

    @Override
    public <T> Future<T> download(DownloadTask<T> task, DownloadCallback<T> callback, int tries) {
        return this.downloadIfNecessary(task, callback, tries);
    }

    @Override
    public void shutdown() {
        try {
            this.upstream.shutdown();
        }
        finally {
            try {
                this.cacheProvider.close();
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Couldn't close cache provider: " + this.cacheProvider, e);
            }
        }
    }

    @Override
    public boolean isShutdown() {
        return this.upstream.isShutdown();
    }

    private <T> Future<T> downloadIfNecessary(DownloadTask<T> task, DownloadCallback<T> callback, int tries) {
        if (task.isCacheable()) {
            T result;
            byte[] cached;
            URI uri = task.getURI();
            String pool = this.resolveCachePool(task.getCachePool());
            if (LOGGER.isLoggable(Level.FINER)) {
                LOGGER.finer(String.format("Resolved the cache pool of [%s]: [%s] -> [%s]", uri, task.getCachePool(), pool));
            }
            if ((cached = this.cacheProvider.get(pool, uri)) == null) {
                return this.submitToUpstream(new CachingDownloadTask<T>(task), callback, tries);
            }
            try {
                result = this.processCache(task, cached);
            }
            catch (Throwable e) {
                this.cacheProvider.remove(pool, uri);
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.log(Level.FINE, String.format("Removed cache [%s] from [%s] because an exception has thrown when applying cache", uri, pool), e);
                }
                return this.submitToUpstream(new CachingDownloadTask<T>(task), callback, tries);
            }
            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER.fine(String.format("Applied cache [%s] from [%s], length=%d", uri, pool, cached.length));
            }
            if (callback != null) {
                callback.done(result);
            }
            return new CompletedFuture<T>(result);
        }
        return this.submitToUpstream(task, callback, tries);
    }

    private <T> Future<T> submitToUpstream(DownloadTask<T> task, DownloadCallback<T> callback, int tries) {
        if (tries == -1) {
            return this.upstream.download(task, callback);
        }
        return this.upstream.download(task, callback, tries);
    }

    private <T> T processCache(DownloadTask<T> task, byte[] cached) throws Exception {
        DownloadSession<T> session = task.createSession(cached.length);
        try {
            session.receiveData(ByteBuffer.wrap(cached));
        }
        catch (Throwable e) {
            session.failed();
            throw e;
        }
        return session.completed();
    }

    private String resolveCachePool(String unresolved) {
        if (unresolved == null) {
            return "org.to2mbn.jmccc.mcdownloader.cache.default";
        }
        while (!this.cacheProvider.hasCache(unresolved)) {
            int lastDot = unresolved.lastIndexOf(46);
            if (lastDot == -1) {
                return "org.to2mbn.jmccc.mcdownloader.cache.default";
            }
            unresolved = unresolved.substring(0, lastDot);
        }
        return unresolved;
    }

    public String toString() {
        return String.format("CachedDownloader [upstream=%s, cacheProvider=%s]", this.upstream, this.cacheProvider);
    }

    private class CachingDownloadTask<T>
    extends DownloadTask<T> {
        private final DownloadTask<T> proxiedTask;

        public CachingDownloadTask(DownloadTask<T> proxiedTask) {
            super(proxiedTask.getURI());
            this.proxiedTask = proxiedTask;
        }

        @Override
        public DownloadSession<T> createSession() throws IOException {
            return new CachingDownloadSession(this.proxiedTask.createSession(), 8192L);
        }

        @Override
        public DownloadSession<T> createSession(long length) throws IOException {
            return new CachingDownloadSession(this.proxiedTask.createSession(length), length);
        }

        private class CachingDownloadSession
        implements DownloadSession<T> {
            private final DownloadSession<T> proxiedSession;
            private SoftReference<ByteArrayOutputStream> bufRef;

            public CachingDownloadSession(DownloadSession<T> proxiedSession, long length) {
                this.proxiedSession = proxiedSession;
                if (length < Integer.MAX_VALUE) {
                    try {
                        this.bufRef = new SoftReference<ByteArrayOutputStream>(new ByteArrayOutputStream((int)length));
                    }
                    catch (OutOfMemoryError e) {
                        this.dropCache();
                    }
                }
            }

            @Override
            public void receiveData(ByteBuffer data) throws IOException {
                byte[] copiedData = new byte[data.remaining()];
                data.get(copiedData);
                this.proxiedSession.receiveData(ByteBuffer.wrap(copiedData));
                if (this.bufRef != null) {
                    try {
                        ByteArrayOutputStream buf = this.bufRef.get();
                        if (buf != null) {
                            buf.write(copiedData);
                        }
                    }
                    catch (OutOfMemoryError e) {
                        this.dropCache();
                    }
                }
            }

            @Override
            public T completed() throws Exception {
                Object result;
                try {
                    result = this.proxiedSession.completed();
                }
                catch (Throwable e) {
                    this.dropCache();
                    throw e;
                }
                this.saveCache();
                return result;
            }

            @Override
            public void failed() throws Exception {
                this.dropCache();
                this.proxiedSession.failed();
            }

            private void dropCache() {
                if (this.bufRef != null) {
                    this.bufRef.clear();
                    this.bufRef = null;
                }
            }

            private void saveCache() {
                if (this.bufRef != null) {
                    try {
                        ByteArrayOutputStream buf = this.bufRef.get();
                        if (buf != null) {
                            byte[] data = buf.toByteArray();
                            URI uri = CachingDownloadTask.this.proxiedTask.getURI();
                            String pool = CachedDownloader.this.resolveCachePool(CachingDownloadTask.this.proxiedTask.getCachePool());
                            CachedDownloader.this.cacheProvider.put(pool, uri, data);
                            if (LOGGER.isLoggable(Level.FINE)) {
                                LOGGER.fine(String.format("Cached [%s] into [%s], length=%d", uri, pool, data.length));
                            }
                        }
                    }
                    catch (OutOfMemoryError e) {
                        this.dropCache();
                    }
                }
            }
        }
    }
}

