package cloud.oneentry.sdk.auth

import cloud.oneentry.sdk.auth.access.AccessService
import cloud.oneentry.sdk.auth.dsl.AuthDataBuilder
import cloud.oneentry.sdk.auth.dsl.SignUpBuilder
import cloud.oneentry.sdk.auth.dto.CreatedUser
import cloud.oneentry.sdk.auth.dto.UserTokenDto
import cloud.oneentry.sdk.auth.dto.UsersAuthProvider
import cloud.oneentry.sdk.core.inject
import cloud.oneentry.sdk.core.query.query
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.request.get
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.http.isSuccess
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put

@Suppress("unused")
object AuthProviderService {
    private val client: HttpClient by inject()
    private val protectedClient: HttpClient by inject("protected")
    private val access: AccessService by inject()

    /**
     * User registration
     *
     * @param marker Text identifier of the authorization provider
     * @param formIdentifier Textual identifier of the authentication provider form
     * @param langCode Locale code in which the user registers
     * @param perform SignUpBuilder dsl for creating a user
     */
    @Throws(Throwable::class)
    suspend fun signUp(
        marker: String,
        formIdentifier: String,
        langCode: String = "en_US",
        perform: SignUpBuilder.() -> Unit
    ): CreatedUser {
        val builder = SignUpBuilder(formIdentifier, langCode)
        builder.perform()

        val response = client.post("users-auth-providers/marker/$marker/users/sign-up") {
            setBody(builder.build())
        }

        return response.body()
    }

    /**
     * Get the activation code for the user
     *
     * @param marker Text identifier of the authorization provider
     * @param userIdentifier Textual identifier of user object (user login)
     * @param eventIdentifier Textual identifier of event object
     */
    @Throws(Throwable::class)
    suspend fun generateCode(
        marker: String,
        userIdentifier: String,
        eventIdentifier: String,
    ) {
        val body = buildJsonObject {
            put("userIdentifier", userIdentifier)
            put("eventIdentifier", eventIdentifier)
        }

        client.post("users-auth-providers/marker/$marker/users/generate-code") {
            setBody(body)
        }
    }


    /**
    * Check user activation code
    *
    * @param marker Text identifier of the authorization provider
    * @param code Service code
    * @param eventIdentifier Textual identifier of user object (user login)
    */
    @Throws(Throwable::class)
    suspend fun checkCode(
        marker: String,
        eventIdentifier: String,
        code: String,
    ) {
        val body = buildJsonObject {
            put("eventIdentifier", eventIdentifier)
            put("code", code)
        }

        client.post("users-auth-providers/marker/$marker/users/check-code") {
            setBody(body)
        }
    }

    @Throws(Throwable::class)
    suspend fun activate(
        marker: String,
        userIdentifier: String,
        code: String,
    ) {
        val body = buildJsonObject {
            put("userIdentifier", userIdentifier)
            put("code", code)
        }
        val response = client.post("users-auth-providers/marker/$marker/users/activate") {
            setBody(body)
        }

        if (!response.status.isSuccess())
            throw IllegalStateException("Failed to activate user")
    }

    /**
    * User authorization
    *
    * @param marker Text identifier of the authorization provider
    * @param perform AuthDataBuilder dsl for user authorization
    */
    @Throws(Throwable::class)
    suspend fun auth(
        marker: String,
        perform: AuthDataBuilder.() -> Unit
    ): UserTokenDto {
        val builder = AuthDataBuilder()
        builder.perform()

        val body = mapOf("authData" to builder.authData)
        val response = client.post("users-auth-providers/marker/$marker/users/auth") {
            setBody(body)
        }

        val tokens: UserTokenDto = response.body()
        access.accessToken = tokens.accessToken
        access.refreshToken = tokens.refreshToken
        access.marker = tokens.authProviderIdentifier

        return tokens
    }

    /**
     * Update user tokens
     *
     * @param marker Text identifier of the authorization provider
     * @param refreshToken Refresh token
     */
    @Throws(Throwable::class)
    suspend fun refresh(
        marker: String,
        refreshToken: String
    ): UserTokenDto {
        val body = buildJsonObject { put("refreshToken", refreshToken) }
        val response = client.post("users-auth-providers/marker/$marker/users/refresh") {
            setBody(body)
        }

        val tokens: UserTokenDto = response.body()
        access.accessToken = tokens.accessToken
        access.refreshToken = tokens.refreshToken
        access.marker = tokens.authProviderIdentifier

        return tokens
    }

    @Throws(Throwable::class)
    suspend fun logout() {
        val marker = access.marker
        val body = mapOf("refreshToken" to access.refreshToken)

        access.refreshToken = null
        access.accessToken = null
        access.marker = null

        protectedClient.post("users-auth-providers/marker/$marker/users/logout") {
            setBody(body)
        }
    }

    @Throws(Throwable::class)
    suspend fun changePassword(
        marker: String,
        userIdentifier: String,
        code: String,
        password: String,
        confirmation: String
    ) {
        val body = cloud.oneentry.sdk.auth.dto.ChangePasswordDto(
            userIdentifier = userIdentifier,
            code = code,
            password1 = password,
            password2 = confirmation
        )

        client.post("users-auth-providers/marker/$marker/users/change-password") {
            setBody(body)
        }
    }

    /**
     * Getting all authorization provider objects
     *
     * @param langCode Language code
     * @param offset Offset parameter for retrieving records
     * @param limit Limit parameter for retrieving records
     */
    @Throws(Throwable::class)
    suspend fun usersAuthProviders(
        offset: Int = 0,
        limit: Int = 30,
        langCode: String = "en_US",
    ): List<UsersAuthProvider> {
        val response = client.get("users-auth-providers") {
            query(
                "offset" to offset,
                "limit" to limit,
                "lang_code" to langCode
            )
        }

        return response.body()
    }

    /**
     * Getting a single authentication provider object by token
     *
     * @param marker Authorization provider object marker
     * @param langCode Language code
     */
    @Throws(Throwable::class)
    suspend fun userAuthProvider(
        marker: String,
        langCode: String = "en_US",
    ): UsersAuthProvider {
        val response = client.get("users-auth-providers/marker/$marker") {
            query("langCode" to langCode)
        }

        return response.body()
    }
}