package cloud.oneentry.sdk.auth.keychain.service

import cloud.oneentry.sdk.auth.keychain.exception.KeychainException
import kotlinx.cinterop.BetaInteropApi
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.UnsafeNumber
import kotlinx.cinterop.alloc
import kotlinx.cinterop.convert
import kotlinx.cinterop.memScoped
import kotlinx.cinterop.ptr
import kotlinx.cinterop.value
import platform.CoreFoundation.CFAutorelease
import platform.CoreFoundation.CFDictionaryAddValue
import platform.CoreFoundation.CFDictionaryCreateMutable
import platform.CoreFoundation.CFMutableDictionaryRef
import platform.CoreFoundation.CFStringRef
import platform.CoreFoundation.CFTypeRef
import platform.CoreFoundation.CFTypeRefVar
import platform.CoreFoundation.kCFBooleanTrue
import platform.Foundation.CFBridgingRelease
import platform.Foundation.CFBridgingRetain
import platform.Foundation.NSBundle
import platform.Foundation.NSData
import platform.Foundation.NSString
import platform.Foundation.NSUTF8StringEncoding
import platform.Foundation.create
import platform.Foundation.dataUsingEncoding
import platform.Security.SecItemAdd
import platform.Security.SecItemCopyMatching
import platform.Security.SecItemDelete
import platform.Security.SecItemUpdate
import platform.Security.errSecDuplicateItem
import platform.Security.errSecItemNotFound
import platform.Security.errSecSuccess
import platform.Security.kSecAttrAccount
import platform.Security.kSecAttrService
import platform.Security.kSecClass
import platform.Security.kSecClassGenericPassword
import platform.Security.kSecMatchLimit
import platform.Security.kSecMatchLimitOne
import platform.Security.kSecReturnData
import platform.Security.kSecValueData

@OptIn(ExperimentalForeignApi::class, UnsafeNumber::class, BetaInteropApi::class)
@Suppress("EXPECT_ACTUAL_CLASSIFIERS_ARE_IN_BETA_WARNING")
actual class KeychainService {
    private fun query(vararg pairs: Pair<CFStringRef?, CFTypeRef?>): CFMutableDictionaryRef? {
        val map = mapOf(*pairs)

        return CFDictionaryCreateMutable(
            null, map.size.convert(), null, null
        ).apply {
            map.forEach { CFDictionaryAddValue(this, it.key, it.value) }
        }.apply {
            CFAutorelease(this)
        }
    }

    actual operator fun get(key: String): String? {
        val bundle = NSBundle.mainBundle.bundleIdentifier ?: "cloud.oneentry.sdk"
        val getQuery = query(
            kSecClass to kSecClassGenericPassword,
            kSecAttrAccount to CFBridgingRetain(key),
            kSecReturnData to kCFBooleanTrue,
            kSecMatchLimit to kSecMatchLimitOne,
            kSecAttrService to CFBridgingRetain(bundle),
        )

        memScoped {
            val result = alloc<CFTypeRefVar>()
            val status = SecItemCopyMatching(getQuery, result.ptr)

            if (status == errSecItemNotFound) {
                return null
            } else if (status != errSecSuccess) {
                throw KeychainException("Error getting item with key $key and status $status")
            } else {
                val data = CFBridgingRelease(result.value) as NSData
                return NSString.create(data, NSUTF8StringEncoding).toString()
            }
        }
    }

    actual operator fun set(key: String, value: String) {
        val string = NSString.create(string = value)
        val data = string.dataUsingEncoding(NSUTF8StringEncoding)
        val bundle = NSBundle.mainBundle.bundleIdentifier ?: "cloud.oneentry.sdk"
        val status = if (this[key] != null) {
            val query = query(
                kSecClass to kSecClassGenericPassword,
                kSecAttrAccount to CFBridgingRetain(key),
                kSecAttrService to CFBridgingRetain(bundle),
            )

            val updateQuery = query(
                kSecAttrAccount to CFBridgingRetain(key),
                kSecValueData to CFBridgingRetain(data),
            )

            SecItemUpdate(query, updateQuery)
        } else {
            val query = query(
                kSecClass to kSecClassGenericPassword,
                kSecAttrAccount to CFBridgingRetain(key),
                kSecValueData to CFBridgingRetain(data),
                kSecAttrService to CFBridgingRetain(bundle),
            )

            SecItemAdd(query, null)
        }

        if (status == errSecDuplicateItem) {
            throw KeychainException("Item with key $key already exists")
        } else if (status != errSecSuccess) {
            throw KeychainException("Error adding item with key $key and status $status")
        }
    }

    actual fun remove(forKey: String) {
        val bundle = NSBundle.mainBundle.bundleIdentifier ?: "cloud.oneentry.sdk"
        val query = query(
            kSecClass to kSecClassGenericPassword,
            kSecAttrAccount to CFBridgingRetain(forKey),
            kSecAttrService to CFBridgingRetain(bundle),
        )

        val status = SecItemDelete(query)

        if (status != errSecSuccess && status != errSecItemNotFound) {
            throw KeychainException("Error deleting item with key $forKey and status $status")
        }
    }
}