/*
 * Copyright (c) 2021 Juraj Jurčo
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
 * associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
 * NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package church.i18n.processing.validation;

import church.i18n.processing.exception.ProcessingException;
import church.i18n.processing.exception.ProcessingExceptionBuilder;
import church.i18n.processing.exception.ProcessingExceptionOptionalBuilder;
import church.i18n.processing.message.ContextInfo;
import church.i18n.processing.message.ContextInfoBuilder;
import church.i18n.processing.message.ContextInfoDefaultBuilder;
import church.i18n.processing.message.ContextValue;
import church.i18n.processing.message.I18nMessage;
import church.i18n.processing.message.ProcessingMessage;
import church.i18n.processing.message.ProcessingMessageBuilder;
import church.i18n.processing.message.ProcessingMessageOptionalBuilder;
import java.util.Optional;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

class ValidatorDefaultBuilder implements ValidatorBuilder {

  @NotNull
  @Override
  public ProcessingMessageBuilder<Optional<ProcessingMessage>> buildProcessingMessage(
      @NotNull final String code, @Nullable final Object... params) {
    return new ProcessingMessageOptionalBuilder(code, params);
  }

  @NotNull
  @Override
  public ProcessingMessageBuilder<Optional<ProcessingMessage>> buildProcessingMessage(
      @NotNull final I18nMessage message) {
    return new ProcessingMessageOptionalBuilder(message);
  }

  @NotNull
  @Override
  public ContextInfoBuilder buildContextInfo(@NotNull final String name) {
    return new ContextInfoDefaultBuilder(name);
  }

  @NotNull
  @Override
  public ProcessingExceptionBuilder buildException(@NotNull final String code,
      @Nullable final Object... params) {
    return new ProcessingExceptionOptionalBuilder(code, params);
  }

  @NotNull
  @Override
  public ProcessingExceptionBuilder buildException(@NotNull final ProcessingMessage message) {
    return new ProcessingExceptionOptionalBuilder(message);
  }

  @NotNull
  @Override
  public ProcessingExceptionBuilder buildException(@NotNull final ProcessingMessage message,
      @NotNull final Throwable cause) {
    return new ProcessingExceptionOptionalBuilder(message, cause);
  }

  @NotNull
  @Override
  public Optional<ContextInfo> createContextInfo(@NotNull final String name,
      @Nullable final ContextValue context) {
    return Optional.ofNullable(new ContextInfoDefaultBuilder(name).withContext(context).build());
  }

  @NotNull
  @Override
  public Optional<ContextInfo> createContextInfo(@NotNull final String name,
      @Nullable final ContextValue context, @Nullable final I18nMessage message) {
    return Optional.ofNullable(new ContextInfoDefaultBuilder(name)
        .withContext(context)
        .withMessage(message)
        .build());
  }

  @NotNull
  @Override
  public Optional<ProcessingException> createException(@NotNull final String code,
      @Nullable final Object... params) {
    return Optional.of(new ProcessingException(code, params));
  }

  @NotNull
  @Override
  public Optional<ProcessingException> createException(@NotNull final ProcessingMessage message) {
    return Optional.of(new ProcessingException(message));
  }

  @NotNull
  @Override
  public Optional<ProcessingException> createException(@NotNull final ProcessingMessage message,
      @NotNull final Throwable cause) {
    return Optional.of(new ProcessingException(message, cause));
  }

  @NotNull
  @Override
  public Optional<ProcessingMessage> createMessage(@NotNull final String code,
      @Nullable final Object... params) {
    return Optional.of(new ProcessingMessage(code, params));
  }

  @NotNull
  @Override
  public Optional<ProcessingMessage> createMessage(@NotNull final I18nMessage message) {
    return Optional.of(new ProcessingMessage(message));
  }

  @Override
  public void throwException(@NotNull final String code,
      @Nullable final Object... params) {
    buildException(code, params).throwException();
  }

  @Override
  public void throwException(@NotNull final ProcessingMessage message) {
    buildException(message).throwException();
  }

  @Override
  public void throwException(@NotNull final ProcessingMessage message,
      @NotNull final Throwable cause) {
    buildException(message, cause).throwException();
  }

  @Override
  public void throwException(@NotNull final ProcessingException exception) {
    throw exception;
  }

  @NotNull
  @Override
  public String toString() {
    return "ValidatorDefaultBuilder{}";
  }
}
