/*
 * Copyright (c) 2021 Juraj Jurčo
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
 * associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
 * NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package church.i18n.processing.exception;

import church.i18n.processing.logger.LogLevel;
import church.i18n.processing.message.ContextInfo;
import church.i18n.processing.message.MessageType;
import church.i18n.processing.message.ProcessingMessage;
import church.i18n.processing.security.policy.SecurityLevel;
import church.i18n.processing.status.DefaultStatus;
import church.i18n.processing.status.Status;
import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Generic exception with enough information but not more for handling exceptional state in the
 * application. It holds information user may need to know what has happened, how to deal with the
 * problem or where to look for a solution.
 */
public class ProcessingException extends RuntimeException
    implements ProcessingInfo, ProcessingExceptionIntermediate<ProcessingException> {

  private static final long serialVersionUID = 1880343548762429830L;

  @NotNull
  private Status status = DefaultStatus.UNKNOWN;
  @NotNull
  private LogLevel logLevel = LogLevel.MESSAGE_TYPE_MAPPING;
  @NotNull
  private ProcessingMessage processingMessage;

  /**
   * Constructor with base code and message parameters.
   *
   * @param code   The {@code code} that may reference message in localization property file. When
   *               no such localized message with {@code code} has been found, the code itself is
   *               used as message.
   * @param params list of parameters needed for construction of localized message. If the last
   *               parameter is {@link Throwable}, it is used as cause of this exception.
   */
  public ProcessingException(@NotNull final String code, @Nullable final Object... params) {
    super(code);
    if (params == null) {
      this.processingMessage = new ProcessingMessage(code);
    } else if (params.length > 0 && params[params.length - 1] instanceof Throwable) {
      initCause((Throwable) params[params.length - 1]);
      Object[] newParams = Arrays.copyOf(params, params.length - 1);
      this.processingMessage = new ProcessingMessage(code, newParams);
    } else {
      this.processingMessage = new ProcessingMessage(code, params);
    }
  }

  /**
   * Constructor with specified message.
   *
   * @param message Error message of the exception.
   */
  public ProcessingException(@NotNull final ProcessingMessage message) {
    super(message.getMessage().getCode());
    this.processingMessage = message;
  }

  /**
   * Constructor with a message and concrete cause of this exception.
   *
   * @param message A message of the exception.
   * @param cause   The throwable that caused this {@link ProcessingException} to get thrown.
   */
  public ProcessingException(@NotNull final ProcessingMessage message,
      @Nullable final Throwable cause) {
    this(message);
    if (cause != null) {
      initCause(cause);
      setStackTrace(cause.getStackTrace());
    }
  }

  @Override
  @NotNull
  public ProcessingException addContextInfo(@Nullable final List<ContextInfo> contextInfo) {
    this.processingMessage = ProcessingMessage
        .withMessage(this.processingMessage)
        .addContextInfo(contextInfo)
        .build();
    return this;
  }

  @Override
  @NotNull
  public ProcessingException addContextInfo(@Nullable final ContextInfo... contextInfo) {
    this.processingMessage = ProcessingMessage
        .withMessage(this.processingMessage)
        .addContextInfo(contextInfo)
        .build();
    return this;
  }

  @Override
  @NotNull
  public ProcessingException withHelpUri(@Nullable final URI helpUri) {
    this.processingMessage = ProcessingMessage
        .withMessage(this.processingMessage)
        .withHelpUri(helpUri)
        .build();
    return this;
  }

  @Override
  @NotNull
  public ProcessingException withHelpUri(@NotNull final String helpUri) {
    this.processingMessage = ProcessingMessage
        .withMessage(this.processingMessage)
        .withHelpUri(helpUri)
        .build();
    return this;
  }

  @NotNull
  @Override
  public ProcessingException withSecurityLevel(@Nullable final SecurityLevel securityLevel) {
    this.processingMessage = ProcessingMessage
        .withMessage(this.processingMessage)
        .withSecurityLevel(securityLevel)
        .build();
    return this;
  }

  @Override
  @NotNull
  public ProcessingException withMessageType(@NotNull final MessageType messageType) {
    this.processingMessage = ProcessingMessage
        .withMessage(this.processingMessage)
        .withMessageType(messageType)
        .build();
    return this;
  }

  @NotNull
  @Override
  public LogLevel getLogLevel() {
    return this.logLevel;
  }

  @NotNull
  @Override
  public ProcessingMessage getProcessingMessage() {
    return this.processingMessage;
  }

  @NotNull
  @Override
  public Status getStatus() {
    return this.status;
  }

  @Override
  @NotNull
  public ProcessingException withLogLevel(@NotNull final LogLevel logLevel) {
    this.logLevel = logLevel;
    return this;
  }

  @Override
  @NotNull
  public ProcessingException withStatus(@NotNull final Status status) {
    this.status = status;
    return this;
  }

  @Override
  public int hashCode() {
    return Objects.hash(this.status, this.logLevel, this.processingMessage);
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj) {
      return true;
    }
    if (!(obj instanceof final ProcessingException that)) {
      return false;
    }
    return this.status.equals(that.status)
        && this.logLevel == that.logLevel
        && Objects.equals(this.processingMessage, that.processingMessage);
  }

  @NotNull
  @Override
  public String toString() {
    return "ProcessingException{" +
        "status=" + this.status +
        ", logLevel=" + this.logLevel +
        ", processingMessage=" + this.processingMessage +
        "} ";
  }
}
