/*
 * Copyright (c) 2021 Juraj Jurčo
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
 * associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
 * NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package church.i18n.processing.message;

import church.i18n.processing.security.policy.SecurityLevel;
import java.util.Objects;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class ContextInfoDefaultBuilder implements ContextInfoBuilder {

  @NotNull
  private final String name;
  @Nullable
  private ContextValue context;
  @Nullable
  private ContextValue help;
  @Nullable
  private I18nMessage message;
  @Nullable
  private SecurityLevel securityLevel;

  public ContextInfoDefaultBuilder(@NotNull final String name) {
    this.name = name;
  }

  @Nullable
  @Override
  public ContextInfo build() {
    return new ContextInfo(this.name, this.context, this.help, this.message, this.securityLevel);
  }

  @NotNull
  @Override
  public ContextInfoBuilder withContext(@Nullable final Object value,
      @Nullable final ValueType valueType) {
    this.context = new ContextValue(value, valueType);
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withContext(@Nullable final Object value,
      @Nullable final String valueType) {
    this.context = new ContextValue(value, valueType);
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withContext(@Nullable final Object value) {
    this.context = new ContextValue(value);
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withContext(@Nullable final ContextValue context) {
    this.context = context;
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withHelp(@Nullable final ContextValue help) {
    this.help = help;
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withHelp(@Nullable final Object help,
      @Nullable final ValueType helpType) {
    this.help = new ContextValue(help, helpType);
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withHelp(@Nullable final Object help,
      @Nullable final String helpType) {
    this.help = new ContextValue(help, helpType);
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withHelp(@Nullable final Object help) {
    this.help = new ContextValue(help);
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withMessage(@Nullable final I18nMessage message) {
    this.message = message;
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withMessage(@Nullable final String code,
      @Nullable final Object... params) {
    if (code != null) {
      this.message = new I18nMessage(code, params);
    }
    return this;
  }

  @NotNull
  @Override
  public ContextInfoBuilder withSecurityLevel(@Nullable final SecurityLevel securityLevel) {
    this.securityLevel = securityLevel;
    return this;
  }

  @Override
  public int hashCode() {
    return Objects.hash(this.name, this.context, this.help, this.message, this.securityLevel);
  }

  @Override
  public boolean equals(@Nullable final Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    ContextInfoDefaultBuilder that = (ContextInfoDefaultBuilder) o;
    return this.name.equals(that.name) &&
        Objects.equals(this.context, that.context) &&
        Objects.equals(this.help, that.help) &&
        Objects.equals(this.message, that.message) &&
        this.securityLevel == that.securityLevel;
  }

  @NotNull
  @Override
  public String toString() {
    return "ContextInfoDefaultBuilder{" +
        "name='" + this.name + '\'' +
        ", context=" + this.context +
        ", help=" + this.help +
        ", message=" + this.message +
        ", securityLevel=" + this.securityLevel +
        '}';
  }
}
