/*******************************************************************************
 * Copyright (c) 2011 Michael Ruflin, Andr Locher, Claudia von Bastian.
 * 
 * This file is part of Tatool.
 * 
 * Tatool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published 
 * by the Free Software Foundation, either version 3 of the License, or 
 * (at your option) any later version.
 * 
 * Tatool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with Tatool. If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ch.tatool.core.display.swing.action;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides a text field as input
 * 
 * @author Andre Locher
 */
public class InputActionPanel extends AbstractActionPanel {

	private static final long serialVersionUID = -7724432302339067931L;

	Logger logger = LoggerFactory.getLogger(InputActionPanel.class);

	public static final int FORMAT_ALL = 0;
	public static final int FORMAT_ONLY_DIGITS = 1;

	/** Creates new form InputActionPanel */
	public InputActionPanel() {
		initComponents();
	}

	/** Call this method when starting the task. */
	public void enableActionPanel() {
		inputTextField.setEnabled(true);
		inputTextField.requestFocusInWindow();
	}

	/** Call this method when stopping the task. */
	public void disableActionPanel() {
		inputTextField.setEnabled(false);
	}

	/** Call this method to clear the text field. */
	public void clearTextField() {
		inputTextField.setText("");
	}

	public void markTextField() {
		inputTextField.selectAll();
	}

	public void setTextDocument(int size, int inputFormat) {
		inputTextField.setDocument(new JTextFieldDocument(size, inputFormat));
	}

	class JTextFieldDocument extends PlainDocument {

		private int limit;
		private int inputFormat = 0;

		// optional uppercase conversion
		private boolean toUppercase = false;

		JTextFieldDocument(int limit, int inputFormat) {
			super();
			this.limit = limit;
			this.inputFormat = inputFormat;
		}

		public void insertString(int offset, String str, AttributeSet attr)
				throws BadLocationException {
			if (str == null)
				return;

			if (inputFormat == FORMAT_ONLY_DIGITS) {
				try {
					Integer.parseInt(str + "0");
				} catch (NumberFormatException e) {
					return;
				}
			}

			if ((getLength() + str.length()) <= limit) {
				if (toUppercase)
					str = str.toUpperCase();
				super.insertString(offset, str, attr);
			}
		}
	}

	/**
	 * This method is called from within the constructor to initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is always
	 * regenerated by the Form Editor.
	 */
	@SuppressWarnings("unchecked")
	// <editor-fold defaultstate="collapsed"
	// desc="Generated Code">//GEN-BEGIN:initComponents
	private void initComponents() {

		inputTextField = new javax.swing.JTextField();

		setBorder(javax.swing.BorderFactory.createEmptyBorder(10, 50, 10, 50));
		setMinimumSize(new java.awt.Dimension(250, 100));
		setPreferredSize(new java.awt.Dimension(250, 100));
		setLayout(new java.awt.BorderLayout());

		inputTextField.setFont(inputTextField.getFont().deriveFont(
				inputTextField.getFont().getSize() + 35f));
		inputTextField.setHorizontalAlignment(javax.swing.JTextField.CENTER);
		inputTextField.setText("");
		inputTextField.setMinimumSize(new java.awt.Dimension(150, 60));
		inputTextField.setPreferredSize(new java.awt.Dimension(150, 60));
		inputTextField.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent evt) {
				inputTextFieldActionPerformed(evt);
			}
		});
		add(inputTextField, java.awt.BorderLayout.CENTER);
	}// </editor-fold>//GEN-END:initComponents

	private void inputTextFieldActionPerformed(java.awt.event.ActionEvent evt) {// GEN-FIRST:event_inputTextFieldActionPerformed
		fireActionTriggered(inputTextField.getText());
	}// GEN-LAST:event_inputTextFieldActionPerformed

	// Variables declaration - do not modify//GEN-BEGIN:variables
	private javax.swing.JTextField inputTextField;
	// End of variables declaration//GEN-END:variables

}
