/*******************************************************************************
 * Copyright (c) 2011 Michael Ruflin, Andr Locher, Claudia von Bastian.
 * 
 * This file is part of Tatool.
 * 
 * Tatool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published 
 * by the Free Software Foundation, either version 3 of the License, or 
 * (at your option) any later version.
 * 
 * Tatool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with Tatool. If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ch.tatool.app.service.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.hibernate.Query;
import org.hibernate.SessionFactory;

import ch.tatool.app.data.ModuleImpl;
import ch.tatool.app.data.ModuleSessionImpl;
import ch.tatool.app.data.TrialImpl;
import ch.tatool.app.data.UserAccountImpl;
import ch.tatool.data.Module;

/**
 * Property access object for module objects.
 * 
 * @author Michael Ruflin
 */
public class ModuleDAO {

	private SessionFactory sessionFactory;

	/**
	 * Get info objects of all available module instances for an account.
	 * 
	 * @param account
	 * @return
	 */

	@SuppressWarnings("unchecked")
	public Set<Module.Info> getModules(final UserAccountImpl account) {
		Query query = sessionFactory
				.getCurrentSession()
				.createQuery(
						"select module.id, module.name FROM ModuleImpl as module where module.accountId = :accountId");
		query.setParameter("accountId", account.getId());
		List<Object[]> results = (List<Object[]>) query.list();
		Set<Module.Info> infos = new TreeSet<Module.Info>();
		for (Object[] result : results) {
			ModuleInfoImpl info = new ModuleInfoImpl();
			info.setAccount(account);
			info.setId((Long) result[0]);
			info.setName((String) result[1]);
			infos.add(info);
		}
		return infos;
	}

	/**
	 * Loads a module object from the database, using the id specified in the
	 * module.
	 * 
	 * If no module with given id exists in the database, a new one is created
	 * 
	 * @param moduleInfo
	 *            the module to load
	 */
	public ModuleImpl loadModule(ModuleInfoImpl moduleInfo) {
		ModuleImpl module = (ModuleImpl) sessionFactory.getCurrentSession()
				.get(ModuleImpl.class, moduleInfo.getId());

		// set the account object manually
		if (!moduleInfo.getAccount().getId().equals(module.getAccountId())) {
			throw new RuntimeException(
					"ModuleInfo and Module object don't belong to each other! Account id mismatch");
		}
		module.setAccount(moduleInfo.getAccount());

		return module;
	}

	/**
	 * Deletes a module
	 * 
	 * @param module
	 *            the module to load
	 */
	public void deleteModule(final ModuleInfoImpl moduleInfo) {
		// PENDING: can't hibernate figure this out on its own?
		// TODO: Test this if removed
		// First delete all trials
		List<?> trialIds = sessionFactory
				.getCurrentSession()
				.createQuery(
						"select id from TrialImpl trial where trial.session.module.id = :id")
				.setParameter("id", moduleInfo.getId()).list();
		for (Object o : trialIds) {
			TrialImpl trial = new TrialImpl();
			sessionFactory.getCurrentSession().load(trial, (Long) o);
			sessionFactory.getCurrentSession().delete(trial);
		}
		sessionFactory.getCurrentSession().flush();

		// then delete the sessions
		List<?> sessionIds = sessionFactory
				.getCurrentSession()
				.createQuery(
						"select id from ModuleSessionImpl session where session.module.id = :id")
				.setParameter("id", moduleInfo.getId()).list();
		for (Object o : sessionIds) {
			ModuleSessionImpl s = new ModuleSessionImpl();
			sessionFactory.getCurrentSession().load(s, (Long) o);
			sessionFactory.getCurrentSession().delete(s);
		}

		// and finally the module itself
		ModuleImpl module = loadModule(moduleInfo);
		sessionFactory.getCurrentSession().delete(module);
	}

	public void addModuleProperties(final ModuleImpl impl,
			final Map<String, String> properties) {

		sessionFactory.getCurrentSession().update(impl);
		Map<String, String> props = impl.getModuleProperties();

		for (String key : properties.keySet()) {
			String value = properties.get(key);
			props.put(key, value);
		}
		sessionFactory.getCurrentSession().save(impl);
	}

	/**
	 * Stores the module object.
	 */
	public void saveModule(ModuleImpl module) {
		sessionFactory.getCurrentSession().saveOrUpdate(module);
	}

	/**
	 * Deletes a module object.
	 */
	public void deleteModule(Module module) {
		sessionFactory.getCurrentSession().delete(module);
	}

	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}
}
