/*******************************************************************************
 * Copyright (c) 2011 Michael Ruflin, Andr Locher, Claudia von Bastian.
 * 
 * This file is part of Tatool.
 * 
 * Tatool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published 
 * by the Free Software Foundation, either version 3 of the License, or 
 * (at your option) any later version.
 * 
 * Tatool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with Tatool. If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ch.tatool.app.export;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import au.com.bytecode.opencsv.CSVWriter;

import ch.tatool.app.data.UserAccountImpl;
import ch.tatool.data.Module;
import ch.tatool.data.DataService;

/**
 * Exports account data into a CSV file.
 * 
 * @author Andr Locher
 */
public class CSVAccountDataExport {

	Logger logger = LoggerFactory.getLogger(CSVAccountDataExport.class);
	
	private List<String> baseProperties;
    private int basePropertiesSize;
    private List<String> accountProperties;
    private int accountPropertiesSize;

	public CSVAccountDataExport(DataService dataService) {
		// this won't change
        initBaseProperties();
        initAccountProperties();
	}

	/**
	 * Exports the account data and stores the results in the provided Writer
	 */
	public File exportData(Module module) {
		// create a writer
        try {
            File tmpFile = File.createTempFile("accountData", "csv");
            FileOutputStream fos = new FileOutputStream(tmpFile, false);
            CSVWriter writer = new CSVWriter(new OutputStreamWriter(fos,"ISO-8859-1"), ';');
  
            // write the csv file header
            List<String> headers = new ArrayList<String>(baseProperties);
            headers.addAll(accountProperties);
            writer.writeNext(headers.toArray(new String[headers.size()]));
   
            // fetch account data
            List<String[]> moduleData = getDataForAccount(module);
            writer.writeAll(moduleData);
            
            // close the writer and return the file
            writer.close();
            return tmpFile;
        } catch (IOException ioe) {
            logger.error("Unable to write csv file", ioe);
            return null;
        }
	}
	
	/** Basic properties which are always written. */
    private void initBaseProperties() {
        baseProperties = new ArrayList<String>();
        
        // account related
        baseProperties.add("Account id");
        baseProperties.add("Account name");
        baseProperties.add("Account password");
        baseProperties.add("Account folder");
       
        basePropertiesSize = baseProperties.size();        
    }
    
    private void initAccountProperties() {
        // add additional headers for account properties
    	accountProperties = new ArrayList<String>();
    	accountProperties.add("PROPERTY_NAME");
    	accountProperties.add("PROPERTY_VALUE");
    	accountPropertiesSize = accountProperties.size();   
    }
    
    private String[] getBaseProperties(Module module) {
    	UserAccountImpl account = (UserAccountImpl) module.getUserAccount();
    	// account related
    	String[] baseProps = new String[4];
    	baseProps[0] = account.getId() + "";
    	baseProps[1] = account.getName();
    	baseProps[2] = account.getPassword();
    	baseProps[3] = account.getFolder().toString();
    	return baseProps;
    }

    private void addToData(List<String[]> data, String[] values) {
        if (values != null) {
            data.add(values);
        }
    }
    
    /** Returns a list with all to be written values for a given account object. */
    private List<String[]> getDataForAccount(Module module) {
        List<String[]> data = new ArrayList<String[]>(basePropertiesSize + accountPropertiesSize);

        // add account properties
        addAccountProperties(module, data);
        
        return data;
    }
    
    /**
     * Adds the account properties to the data object
     */
    private void addAccountProperties(Module module, List<String[]> data) {
    	Map<String, String> account = module.getUserAccount().getProperties();
    	List<String> accountProps = null;
    	
    	Set<Map.Entry<String, String>> entries = account.entrySet();
		for (Map.Entry<String, String> entry : entries) {
			accountProps = new ArrayList<String>();
        	accountProps.addAll(Arrays.asList(getBaseProperties(module)));
        	accountProps.add(entry.getKey());
        	accountProps.add(entry.getValue());
        	addToData(data, accountProps.toArray(new String[accountProps.size()]));
		}
    }
}
