// Copyright 2022-2023 Buf Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package build.buf.connect.extensions

import build.buf.connect.Method
import build.buf.connect.Serializable
import com.squareup.moshi.Moshi
import com.squareup.wire.Message
import com.squareup.wire.WireJsonAdapterFactory
import kotlin.reflect.KClass
import kotlin.reflect.full.createInstance
import okio.Buffer
import okio.BufferedSource
import okio.ByteString.Companion.encodeUtf8

fun <Input : Message<Input, *>, Output : Message<Output, *>> method(
    path: String,
    request: KClass<Input>,
    response: KClass<Output>
): Method<Input, Output> {
    return Method(path, WireProtoAdapter(request), WireProtoAdapter(response))
}

/**
 * Adapter for Connect to use square/wire's runtime for
 * deserializing and serializing square/wire based data
 * types.
 */
private class WireProtoAdapter<E : Message<E, *>>(
    clazz: KClass<E>
) : Serializable<E> {
    val adapter by lazy { clazz.createInstance().adapter }

    private val jsonDecoder = Moshi.Builder()
        .add(WireJsonAdapterFactory())
        .build()

    override fun toBinary(e: E): Buffer {
        return Buffer().write(adapter.encode(e))
    }

    override fun fromBinary(source: BufferedSource): E {
        return adapter.decode(source)
    }

    override fun fromJson(source: BufferedSource): E {
        return jsonDecoder.adapter(adapter.type!!.java as Class<E>).fromJson(source) ?: adapter.decode(ByteArray(0))
    }

    override fun toJson(e: E): Buffer {
        val jsonString = jsonDecoder.adapter(adapter.type!!.java as Class<E>).toJson(e)
        return Buffer().write(jsonString.encodeUtf8())
    }
}
