/*
 * Copyright 2022 David Gregory
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.github.davidgregory084

import sbt.librarymanagement.CrossVersion

import scala.util.Try

class TpolecatPlugin(
    // The default mode to use for configuring scalac options via the sbt-tpolecat plugin.
    val tpolecatDefaultOptionsMode: OptionsMode = CiMode,
    // The set of scalac options that will be excluded.
    val tpolecatExcludeOptions: Set[ScalacOption] = Set.empty
) {
  // The environment variable to use to enable the sbt-tpolecat development mode.
  val tpolecatDevModeEnvVar: String = "SBT_TPOLECAT_DEV"
  // The environment variable to use to enable the sbt-tpolecat continuous integration mode.
  val tpolecatCiModeEnvVar: String = "SBT_TPOLECAT_CI"
  // The environment variable to use to enable the sbt-tpolecat release mode.
  val tpolecatReleaseModeEnvVar: String = "SBT_TPOLECAT_RELEASE"

  // The mode to use for configuring scalac options via the sbt-tpolecat plugin.
  def tpolecatOptionsMode: OptionsMode =
    if (sys.env.contains(tpolecatReleaseModeEnvVar)) ReleaseMode
    else if (sys.env.contains(tpolecatCiModeEnvVar)) CiMode
    else if (sys.env.contains(tpolecatDevModeEnvVar)) DevMode
    else tpolecatDefaultOptionsMode

  // The set of scalac options that will be applied by the sbt-tpolecat plugin in the development mode.
  def tpolecatDevModeOptions: Set[ScalacOption] =
    ScalacOptions.default

  // The set of scalac options that will be applied by the sbt-tpolecat plugin in the continuous integration mode.
  def tpolecatCiModeOptions: Set[ScalacOption] =
    tpolecatDevModeOptions + ScalacOptions.fatalWarnings

  // The set of scalac options that will be applied by the sbt-tpolecat plugin in the release mode.
  def tpolecatReleaseModeOptions: Set[ScalacOption] =
    tpolecatCiModeOptions + ScalacOptions.optimizerMethodLocal

  // The set of scalac options that will be applied by the sbt-tpolecat plugin.
  def tpolecatScalacOptions: Set[ScalacOption] =
    tpolecatOptionsMode match {
      case DevMode     => tpolecatDevModeOptions
      case CiMode      => tpolecatCiModeOptions
      case ReleaseMode => tpolecatReleaseModeOptions
    }

  object ScalacOptions extends ScalacOptions

  def scalacOptionsFor(
      version: String,
      modeScalacOptions: Set[ScalacOption]
  ): Seq[String] = {
    val supportedOptions = (CrossVersion.partialVersion(version), version.split('.')) match {
      case (Some((0, _)), _) => // dotty prereleases use 0 as major version
        modeScalacOptions
          .filter(_.isSupported(ScalaVersion.V3_0_0)) // treat dotty prereleases as 3.0.0
      case (Some((maj, min)), Array(maj2, min2, patch)) if maj.toString == maj2 && min.toString == min2 =>
        modeScalacOptions
          .filter(_.isSupported(ScalaVersion(maj, min, Try(patch.toLong).getOrElse(0))))
      case (Some((maj, min)), _) =>
        modeScalacOptions
          .filter(_.isSupported(ScalaVersion(maj, min, 0)))
      case (None, _) =>
        Set.empty[ScalacOption]
    }

    supportedOptions.toList.flatMap(opt => opt.option :: opt.args)
  }

  def scalacOptions(scalaVersion: String): Seq[String] = {
    val pluginOptions = tpolecatScalacOptions
    val pluginExcludes = tpolecatExcludeOptions
    val selectedOptions = pluginOptions.diff(pluginExcludes)
    scalacOptionsFor(scalaVersion, selectedOptions)
  }
}
