package com.hp.message.utils;

import com.hp.message.Constant;
import com.hp.message.domain.AppChannel;
import com.hp.message.enums.AppChannelType;
import com.hp.message.enums.EmqxTopicType;
import org.springframework.util.StringUtils;

/**
 * @author 尚肖磊
 *  2021-04-25 10:08
 *  sdk mqtt通信topic助手类
 */
public class MsgTopicUtil {

    /**
     * 获取topic builder对象
     *
     * @param emqxTopicType emqx通道类型
     * @param appId           应用id
     * @return
     */
    public static StringBuilder getTopicBuilder(EmqxTopicType emqxTopicType, String appId) {
        StringBuilder sb = new StringBuilder();
        sb.append(Constant.APP_TOPIC_PREFIX).append("/");
        sb.append(emqxTopicType.getChannelId()).append("/");
        sb.append(appId).append("/");
        return sb;
    }

    /**
     * 获取项目通信发送主题
     *
     * @param appId     应用id
     * @param projectId 项目id
     * @return
     */
    public static String getProjectPushTopic(String appId, String projectId) {
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_PUH_CHANNEL, appId);
        sb.append(AppChannelType.APP_PROJECT_CHANNEL.getChannelId()).append("/");
        sb.append(projectId);
        return sb.toString();
    }

    /**
     * 获取分组通信发送主题
     *
     * @param appId   应用id
     * @param groupId 项目分组id
     * @return
     */
    public static String getGroupPushTopic(String appId, String groupId) {
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_PUH_CHANNEL, appId);
        sb.append(AppChannelType.APP_GROUP_CHANNEL.getChannelId()).append("/");
        sb.append(groupId);
        return sb.toString();
    }

    /**
     * 获取设备通信发送主题
     *
     * @param appId   应用id
     * @param equiSno 设备标识
     * @return
     */
    public static String getEquiPushTopic(String appId, String equiSno) {
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_PUH_CHANNEL, appId);
        sb.append(AppChannelType.APP_ID_CHANNEL.getChannelId()).append("/");
        sb.append(equiSno);
        return sb.toString();
    }

    /**
     * 获取项目通信接收主题
     *
     * @param appId     应用id
     * @param projectId 项目id
     * @return
     */
    public static String getProjectSubTopic(String appId, String projectId) {
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_SUB_CHANNEL, appId);
        sb.append(AppChannelType.APP_PROJECT_CHANNEL.getChannelId()).append("/");
        sb.append(projectId);
        return sb.toString();
    }

    /**
     * 获取分组通信接收主题
     *
     * @param appId   应用id
     * @param groupId 项目分组id
     * @return
     */
    public static String getGroupSubTopic(String appId, String groupId) {
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_SUB_CHANNEL, appId);
        sb.append(AppChannelType.APP_GROUP_CHANNEL.getChannelId()).append("/");
        sb.append(groupId);
        return sb.toString();
    }

    /**
     * 获取设备通信接收主题
     *
     * @param appId   应用id
     * @param equiSno 设备标识
     * @return
     */
    public static String getEquiSubTopic(String appId, String equiSno) {
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_SUB_CHANNEL, appId);
        sb.append(AppChannelType.APP_ID_CHANNEL.getChannelId()).append("/");
        sb.append(equiSno);
        return sb.toString();
    }

    /**
     * 获取APP通信发送主题
     *
     * @param appId   应用id
     * @return
     */
    public static String getAppPushTopic(String appId) {
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_PUH_CHANNEL, appId);
        sb.append(AppChannelType.APP_CHANNEL.getChannelId()).append("/");
        sb.append(appId);
        return sb.toString();
    }

    /**
     * 获取APP通信接收主题
     *
     * @param appId   应用id
     * @return
     */
    public static String getAppSubTopic(String appId) {
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_SUB_CHANNEL, appId);
        sb.append(AppChannelType.APP_CHANNEL.getChannelId()).append("/");
        sb.append(appId);
        return sb.toString();
    }

    /**
     * 获取应用监听主题
     *
     * @param appId 应用id
     * @return
     */
    public static String getAppDefaultSubTopic(String appId){
        StringBuilder sb = getTopicBuilder(EmqxTopicType.EMQX_APP_SUB_CHANNEL, appId);
        sb.append("#");
        return sb.toString();
    }

    /**
     * 解析接收消息的渠道
     *
     * @param topic 消息主题
     * @return
     */
    public static AppChannel getAppChannel(String topic) throws Exception {
        if (StringUtils.isEmpty(topic)){
            throw new Exception("topic is empty");
        }
        if (!topic.contains("/")){
            throw new Exception("topic not contain \"/\"");
        }
        String[] topicArray = topic.split("/");
        if (5 != topicArray.length){
            throw new Exception("topic format error");
        }
        AppChannel appChannel = new AppChannel();
        appChannel.setAppId(topicArray[2]);
        Integer appChannelId = Integer.valueOf(topicArray[3]);
        AppChannelType channelType
                = AppChannelType.getAppChannelTypeByChannelId(appChannelId);
        appChannel.setChannelType(channelType);
        appChannel.setChannelValue(topicArray[4]);
        return appChannel;
    }



}
