package com.hp.message.service.common;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.concurrent.*;

/**
 * @author 尚肖磊
 *  2021-03-26 14:49
 *  执行异步任务服务类
 */

@Slf4j
@Service
public class AsyncCallService {

    /**
     * 异步任务线程池对象
     */
    private ScheduledThreadPoolExecutor scheduledPoolExecutor;

    /**
     * 异步线程池初始化
     */
    @PostConstruct
    public void initScheduledThreadPool() {
        if (scheduledPoolExecutor == null) {
            ThreadFactory namedThreadFactory = Executors.defaultThreadFactory();
            scheduledPoolExecutor = new ScheduledThreadPoolExecutor(4, namedThreadFactory);
        }
    }

    /**
     * 异步线程池销毁
     */
    @PreDestroy
    public void closeScheduledThreadPool() {
        if (scheduledPoolExecutor != null) {
            scheduledPoolExecutor.shutdown();
        }
    }

    /**
     * 异步处理任务
     *
     * @param runnable
     */
    public void handleCallable(Runnable runnable) {
        scheduledPoolExecutor.execute(runnable);
    }

    /**
     * 异步处理任务
     *
     * @param callback
     */
    public Future handleCallable(Callable callback) {
        return scheduledPoolExecutor.submit(callback);
    }

    /**
     * 达到给定的延时时间后，执行任务
     *
     * @param runnable 任务执行对象
     * @param delay
     * @param unit     延时单位
     */
    public ScheduledFuture addDelayTask(Runnable runnable, long delay, TimeUnit unit) {
        return scheduledPoolExecutor.schedule(runnable, delay, unit);
    }

    /**
     * 达到给定的延时时间后，执行任务
     * <p>
     * 返回处理结果
     *
     * @param callable  任务执行对象
     * @param delay
     * @param unit      延时单位
     * @return 返回
     */
    public <T> ScheduledFuture<T> addDelayTask(Callable<T> callable, long delay, TimeUnit unit){
        return scheduledPoolExecutor.schedule(callable, delay, unit);
    }

}
