package com.hp.message.service;

import com.hp.message.domain.EmqxDataMsg;
import com.hp.message.enums.MqMsgResultType;
import com.hp.message.enums.MqMsgType;
import com.hp.message.event.EmqxSendMsgEvent;
import com.hp.message.interfaces.IFuncCallBack;
import com.hp.message.interfaces.ISdkMsgCallBack;
import com.hp.message.interfaces.ISdkSendMsgService;
import com.hp.message.property.AppProperty;
import com.hp.message.service.common.InnerEventService;
import com.hp.message.service.common.SendMsgCallBackService;
import com.hp.message.utils.MsgTopicUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;

/**
 * @author 尚肖磊
 *  2021-06-25 14:35
 *  sdk发送消息服务类
 */
@Slf4j
public class SdkSendMsgService implements ISdkSendMsgService {

    @Lazy
    @Autowired
    private AppProperty appProperty;
    @Lazy
    @Autowired
    private InnerEventService innerEventService;
    @Lazy
    @Autowired
    private SendMsgCallBackService sendMsgCallBackService;

    /**
     * 向单个设备发送byte[]消息
     *
     * @param equiSno 设备标识
     * @param msgData 字符串类数据
     * @return
     */
    @Override
    public boolean sendMsgToEqui(String equiSno, byte[] msgData) {
        try {
            String msgTopic = MsgTopicUtil.getEquiPushTopic(appProperty.getAppId(), equiSno);
            EmqxDataMsg emqxDataMsg = getEquiEquiDataMsg(equiSno, msgData);
            EmqxSendMsgEvent emqxSendMsgEvent = new EmqxSendMsgEvent(this, emqxDataMsg, msgTopic);
            return sendEmqxMsg(emqxSendMsgEvent);
        } catch (Exception ex) {
            log.error("send msg exception", ex);
            return false;
        }
    }

    /**
     * 向单个设备异步发送byte[]消息
     *
     * @param equiSno      设备标识
     * @param msgData      字符串类数据
     * @param funcCallBack 消息回调
     */
    @Override
    public void sendMsgToEqui(String equiSno, byte[] msgData, IFuncCallBack<String, String> funcCallBack) {
        String msgTopic = MsgTopicUtil.getEquiPushTopic(appProperty.getAppId(), equiSno);
        EmqxDataMsg emqxDataMsg = getEquiEquiDataMsg(equiSno, msgData);
        try {
            if (funcCallBack != null) {
                EmqxDataMsg.RespConfigBean respConfig = new EmqxDataMsg.RespConfigBean();
                respConfig.setRespTimeout(1);
                respConfig.setRespTopic(MsgTopicUtil.getEquiSubTopic(appProperty.getAppId(), equiSno));
                emqxDataMsg.setRespConfig(respConfig);
                // 配置超时检测
                ISdkMsgCallBack tempCallBack = new ISdkMsgCallBack<EmqxDataMsg, EmqxDataMsg>() {
                    @Override
                    public void onSuccess(EmqxDataMsg respDataMsg, EmqxDataMsg reqDataMsg) {
                        funcCallBack.success(respDataMsg.getEquiSno());
                    }

                    @Override
                    public void onError(MqMsgResultType resultType, EmqxDataMsg reqDataMsg) {
                        funcCallBack.fail(resultType, reqDataMsg.getEquiSno());
                    }

                    @Override
                    public void onTimeout(EmqxDataMsg reqDataMsg) {
                        funcCallBack.fail(MqMsgResultType.TIMEOUT, reqDataMsg.getEquiSno());
                    }
                };
                if (!sendMsgCallBackService.addMsgCallbackWrapper(emqxDataMsg, tempCallBack)) {
                    throw new Exception("add msg callback error");
                }
            }
            // 发送消息
            EmqxSendMsgEvent emqxSendMsgEvent = new EmqxSendMsgEvent(this, emqxDataMsg, msgTopic);
            sendEmqxMsg(emqxSendMsgEvent);
        } catch (Exception ex) {
            if (funcCallBack != null) {
                //移除消息回调关系
                sendMsgCallBackService.removeMsgCallBack(emqxDataMsg);
                funcCallBack.fail(MqMsgResultType.MSG_SEND_EXCEPTION, emqxDataMsg.getEquiSno());
            }
            log.error("send msg exception", ex);
        }
    }

    /**
     * 获取设备消息对象
     *
     * @param equiSno 设备sno
     * @param msgData 发送数据
     * @return
     */
    private EmqxDataMsg getEquiEquiDataMsg(String equiSno, byte[] msgData) {
        EmqxDataMsg emqxDataMsg = new EmqxDataMsg();
        emqxDataMsg.setEquiSno(equiSno);
        emqxDataMsg.setMsgType(MqMsgType.EMQX_SDK_PUSH_DATA_REQ);
        emqxDataMsg.setTransferData(msgData);
        return emqxDataMsg;
    }

    /**
     * 发布 发送emqx消息事件
     *
     * @param emqxSendMsgEvent 发送消息对象
     * @return
     */
    private boolean sendEmqxMsg(EmqxSendMsgEvent emqxSendMsgEvent) {
        return innerEventService.publishEvent(emqxSendMsgEvent);
    }

}
