package com.hp.message.interfaces;

import com.github.lianjiatech.retrofit.spring.boot.annotation.RetrofitClient;
import com.github.lianjiatech.retrofit.spring.boot.interceptor.LogLevel;
import com.github.lianjiatech.retrofit.spring.boot.interceptor.LogStrategy;
import com.hp.message.annotation.ApiSign;
import com.hp.message.convert.FastJsonConverterFactory;
import com.hp.message.domain.ApiResp;
import com.hp.message.domain.param.AddEquiParam;
import com.hp.message.domain.result.*;
import com.hp.message.property.ApiProperty;
import retrofit2.http.*;

import java.util.List;

/**
 * @author 尚肖磊
 *  2021-06-22 17:01
 *  平台API接口类定义
 */
@RetrofitClient(
        baseUrl = ApiProperty.API_HOST,
        converterFactories = FastJsonConverterFactory.class,
        logLevel = LogLevel.INFO, logStrategy = LogStrategy.BASIC)
@ApiSign(include = {"/api/sdk/**"})
public interface IRetrofitApiService {

    /**
     * 新增项目接口
     *
     * @param projectName 项目名称
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/project/addProject")
    ApiResp addProject(@Field("projectName") String projectName,
                       @Field("projectDesc") String projectDesc);

    /**
     * 修改项目名称接口
     *
     * @param projectId   项目id
     * @param projectName 修改后项目名称
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/project/alterProject")
    ApiResp alterProject(@Field("projectId") int projectId,
                         @Field("projectName") String projectName);

    /**
     * 删除项目
     *
     * @param projectId 项目id
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/project/removeProjectById")
    ApiResp removeProjectById(@Field("projectId") int projectId);

    /**
     * 获取应用 绑定项目列表接口
     *
     * @return
     */
    @GET("api/sdk/project/getAppProjectList")
    ApiResp<List<ProjectResult>> getAppProjectList();

    /**
     * 修改项目透传状态接口
     *
     * @param projectId      项目id
     * @param enableTransfer 透传状态 true 开启 false 关闭
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/project/alterProjectTransferStatus")
    ApiResp alterProjectTransferStatus(@Field("projectId") int projectId,
                                       @Field("enableTransfer") boolean enableTransfer);

    /**
     * 解绑应用与项目接口
     *
     * @param projectId 项目id
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/project/unbindProject")
    ApiResp unbindProject(@Field("projectId") int projectId);


    /**
     * 获取分组列表接口
     *
     * @param projectId 项目id
     * @param groupName 查询分组名称(模糊匹配)
     * @return
     */
    @GET("api/sdk/group/getGroupList")
    ApiResp<List<GroupResult>> getGroupList(@Query("projectId") int projectId,
                                            @Query("groupName") String groupName);

    /**
     * 创建项目分组接口
     *
     * @param projectId 项目id
     * @param groupName 项目分组名称
     * @param groupSort 项目分组排序值
     * @param groupDesc 项目分组描述
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/group/addGroup")
    ApiResp addGroup(@Field("projectId") int projectId,
                     @Field("groupName") String groupName,
                     @Field("groupSort") int groupSort,
                     @Field("groupDesc") String groupDesc);

    /**
     * 修改项目分组接口
     *
     * @param groupId   项目分组id
     * @param groupName 项目分组名称
     * @param groupSort 项目分组排序值
     * @param groupDesc 项目分组描述
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/group/alterGroup")
    ApiResp alterGroup(@Field("groupId") int groupId,
                       @Field("groupName") String groupName,
                       @Field("groupSort") int groupSort,
                       @Field("groupDesc") String groupDesc);

    /**
     * 删除项目分组
     *
     * @param groupId 项目分组id
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/group/removeGroupById")
    ApiResp removeGroupById(@Field("groupId") int groupId);


    /**
     * 获取设备详情接口
     *
     * @param equiSno 设备sno
     * @return
     */
    @GET("api/sdk/equi/getEquiInfoBySno")
    ApiResp<EquiResult> getEquiInfo(@Query("equiSno") String equiSno);

    /**
     * 获取设备列表接口
     *
     * @param projectId      项目id
     * @param groupId        项目分组id
     * @param equiStatus     设备状态(状态接口获取)
     * @param dataHandleType 数据处理方式(数据处理方式接口获取)
     * @return
     */
    @GET("api/sdk/equi/getEquiList")
    ApiResp<List<EquiResult>> getEquiList(@Query("projectId") int projectId,
                                          @Query("groupId") int groupId,
                                          @Query("equiStatus") int equiStatus,
                                          @Query("dataHandleType") String dataHandleType);

    /**
     * 获取设备状态列表接口
     *
     * @return
     */
    @GET("api/sdk/equi/getEquiWorkStatusList")
    ApiResp<List<StatusResult>> getEquiWorkStatusList();

    /**
     * 添加设备
     *
     * @param addEquiParam 注册设备JSON对象
     * @return
     */
    @POST("api/sdk/equi/addEqui")
    ApiResp<AddEquiResult> addEqui(@Body AddEquiParam addEquiParam);

    /**
     * 批量添加设备
     *
     * @param addEquiParamList 注册设备JSON对象列表
     * @return
     */
    @POST("api/sdk/equi/addEquiByList")
    ApiResp<List<AddEquiResult>> addEquiByList(@Body List<AddEquiParam> addEquiParamList);

    /**
     * 修改设备信息接口
     *
     * @param equiSno      设备sno
     * @param projectId    项目id
     * @param groupId      分组id
     * @param equiName     设备名称
     * @param equiPassword 设备通讯密码
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/equi/alterEquiBySno")
    ApiResp alterEquiBySno(@Field("equiSno") String equiSno,
                           @Field("projectId") int projectId,
                           @Field("groupId") int groupId,
                           @Field("equiName") String equiName,
                           @Field("equiPassword") String equiPassword);

    /**
     * 删除设备接口
     *
     * @param equiSno 设备sno
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/equi/removeEquiBySno")
    ApiResp removeEquiBySno(@Field("equiSno") String equiSno);

    /**
     * 批量删除设备接口
     *
     * @param equiSnos 设备ids(格式 1,2,3,4)
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/equi/removeEquiBySnos")
    ApiResp removeEquiBySnos(@Field("equiSnos") String equiSnos);


    /**
     * 获取设备详情
     *
     * @param equiSno
     * @return
     */
    @FormUrlEncoded
    @POST("api/sdk/equi/getEquiInfoBySno")
    ApiResp getEquiInfoBySno(@Field("equiSno") String equiSno);
}
