package com.hp.message.interceptor;

import com.github.lianjiatech.retrofit.spring.boot.interceptor.BasePathMatchInterceptor;
import com.hp.message.Constant;
import com.hp.message.property.AppProperty;
import com.hp.message.utils.Md5Util;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import okio.Buffer;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.*;

/**
 * @author 尚肖磊
 *  2021-06-02 16:12
 *  api统一签名注解 解释器
 */
@Slf4j
public class ApiSignInterceptor extends BasePathMatchInterceptor {

    private AppProperty appProperty;

    public ApiSignInterceptor(AppProperty appProperty) {
        this.appProperty = appProperty;
    }

    @Override
    protected Response doIntercept(Chain chain) throws IOException {
        Request request = chain.request();
        request = request
                .newBuilder()
                .addHeader(Constant.API_HEADER_APPID_KEY, appProperty.getAppId())
                .build();
        // 根据请求类型 增加公共参数及签名
        if (request.method().equalsIgnoreCase("GET")) {
            request = signGetParams(request);
        } else if (request.method().equalsIgnoreCase("POST")) {
            request = signPostParams(request);
        }
        return chain.proceed(request);
    }

    /**
     * post请求签名
     *
     * @param request 请求对象
     * @return
     */
    private Request signPostParams(Request request) throws IOException {
        // 仅支持FormBody类型
        if (request.body() instanceof FormBody) {
            FormBody formBody = (FormBody) request.body();
            // 添加原始FormBody参数
            FormBody.Builder bodyBuilder = new FormBody.Builder();
            for (int i = 0; i < formBody.size(); i++) {
                bodyBuilder.addEncoded(formBody.encodedName(i), formBody.encodedValue(i));
            }
            // 添加公共参数
            formBody = bodyBuilder
                    .addEncoded(Constant.API_PARAM_TIMESTAMP_NAME, String.valueOf(System.currentTimeMillis()))
                    .build();
            // 构造参数map
            Map<String, String> bodyMap = new HashMap<>();
            for (int i = 0; i < formBody.size(); i++) {
                bodyMap.put(formBody.name(i), formBody.value(i));
            }
            // 构造签名字符串
            String siqnStr = formatApiParam(bodyMap, true, false);
            siqnStr = siqnStr + "&" + Constant.API_PARAM_SIGN_NAME + "=" + appProperty.getAppSecret();
            // 签名
            String sign = Md5Util.getStrMd5Code(siqnStr);
            request = request.newBuilder()
                    .addHeader(Constant.API_HEADER_SIGN_KEY, sign)
                    .post(formBody).build();
        } else if (request.body().contentType().subtype().equalsIgnoreCase("json")) {
            String jsonBody = requestBodyToString(request.body());
            // 构造签名字符串
            String siqnStr = jsonBody + "&" + Constant.API_PARAM_SIGN_NAME + "=" + appProperty.getAppSecret();
            // 签名
            String sign = Md5Util.getStrMd5Code(siqnStr);
            request = request.newBuilder()
                    .addHeader(Constant.API_HEADER_SIGN_KEY, sign)
                    .build();
        }
        return request;
    }

    /**
     * get请求签名
     *
     * @param request 请求对象
     * @return
     */
    private Request signGetParams(Request request) {
        //添加公共参数
        HttpUrl httpUrl = request.url()
                .newBuilder()
                .addQueryParameter(Constant.API_PARAM_TIMESTAMP_NAME, String.valueOf(System.currentTimeMillis()))
                .build();
        // 获取参数名列表
        Set<String> paramNameSet = httpUrl.queryParameterNames();
        // 构造参数map
        Map<String, String> paramMap = new HashMap<>();
        for (String paramName : paramNameSet) {
            paramMap.put(paramName, httpUrl.queryParameter(paramName));
        }
        // 构造签名字符串
        String siqnStr = formatApiParam(paramMap, true, false);
        // 签名
        siqnStr = siqnStr + "&" + Constant.API_PARAM_SIGN_NAME + "=" + appProperty.getAppSecret();
        String sign = Md5Util.getStrMd5Code(siqnStr);
        request = request.newBuilder()
                .url(httpUrl)
                .addHeader(Constant.API_HEADER_SIGN_KEY, sign)
                .build();
        return request;
    }

    /**
     * 格式化签名参数
     *
     * @param param       参数map
     * @param isUrlEncode 参数值是否进行urlEncode
     * @param toLowerCase 参数名及值是否转化小写
     * @return
     */
    private String formatApiParam(Map<String, String> param, boolean isUrlEncode, boolean toLowerCase) {
        String paramSginStr = "";
        Map<String, String> map = param;
        try {
            List<Map.Entry<String, String>> itmes = new ArrayList<>(map.entrySet());
            //对所有传入的参数名按字母从小到大排序
            //可通过实现Comparator接口的compare方法来完成自定义排序
            Collections.sort(itmes, Comparator.comparing(Map.Entry::getKey));
            //构造URL 键值对的形式
            StringBuilder sb = new StringBuilder();
            for (Map.Entry<String, String> item : itmes) {
                if (!StringUtils.isEmpty(item.getKey()) && !StringUtils.isEmpty(item.getValue())) {
                    String key = item.getKey();
                    String val = item.getValue();
                    if (isUrlEncode) {
                        val = URLEncoder.encode(val, "utf-8");
                    }
                    if (toLowerCase) {
                        sb.append(key.toLowerCase() + "=" + val.toLowerCase());
                    } else {
                        sb.append(key + "=" + val);
                    }
                    sb.append("&");
                }
            }
            paramSginStr = sb.toString().trim();
            if (paramSginStr.endsWith("&")) {
                paramSginStr = paramSginStr.substring(0, paramSginStr.length() - 1);
            }
            return paramSginStr;
        } catch (Exception ex) {
            return "";
        }
    }

    /**
     * 获取post json类型请求对象
     *
     * @param requestBody post请求body
     * @return
     * @throws IOException
     */
    private static String requestBodyToString(RequestBody requestBody) throws IOException {
        Buffer buffer = new Buffer();
        requestBody.writeTo(buffer);
        return buffer.readUtf8();
    }

}
