package com.hp.message.handler;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.hp.message.domain.EmqxClient;
import com.hp.message.domain.EmqxDataMsg;
import com.hp.message.domain.EmqxResult;
import com.hp.message.enums.EmqxQosType;
import com.hp.message.event.EmqxSendMsgEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

/**
 * @author 尚肖磊
 *  2021-06-25 11:18
 *  emqx消息发送事件服务类
 */
@Slf4j
@Component
public class SendMessageHandler {

    @Lazy
    @Autowired
    private EmqxClient emqxClient;

    /**
     * 处理发送emqx消息事件
     *
     * @param event 消息对象
     */
    @Async
    @EventListener
    public void onApplicationEvent(EmqxSendMsgEvent event) {
        if (event.getQosType() != null) {
            sendMessageToMqtt(event.getEmqxTopic(), event.getEventData(), event.getQosType());
        } else {
            sendMessageToMqtt(event.getEmqxTopic(), event.getEventData());
        }
    }

    /**
     * 发送mqtt消息
     *
     * @param topic       发送主题
     * @param emqxDataMsg 发送消息对象
     */
    private void sendMessageToMqtt(String topic, EmqxDataMsg emqxDataMsg) {
        sendMessageToMqtt(topic, emqxDataMsg, EmqxQosType.QOS_0);
    }

    /**
     * 发送mqtt消息
     *
     * @param topic       发送主题
     * @param emqxDataMsg 发送消息对象
     * @param qos         消息质量 0 最多一次 1 至少一次 2 仅一次
     */
    private void sendMessageToMqtt(String topic, EmqxDataMsg emqxDataMsg, EmqxQosType qos) {
        EmqxResult emqxResult = emqxClient.pushMqttMessage(topic, JSON.toJSONBytes(emqxDataMsg, SerializerFeature.DisableCircularReferenceDetect), qos);
        if (emqxResult.isSuccess()) {
            log.debug("send msg topic {} data {} qos {}", topic, JSON.toJSONString(emqxDataMsg), qos);
        } else {
            log.error("send msg topic {} data {} qos {} error {}", topic, JSON.toJSONString(emqxDataMsg), qos, emqxResult.getErrorMsg());
        }
    }

}
