package com.hp.message.handler;

import com.hp.message.domain.EmqxDataMsg;
import com.hp.message.domain.MsgCallBackWrapper;
import com.hp.message.enums.EquiLogoutType;
import com.hp.message.enums.MqMsgProgressedType;
import com.hp.message.enums.MqMsgResultType;
import com.hp.message.enums.MqMsgType;
import com.hp.message.event.EmqxReceMsgEvent;
import com.hp.message.event.EmqxSendMsgEvent;
import com.hp.message.interfaces.ISdkReceMsgService;
import com.hp.message.service.common.AsyncCallService;
import com.hp.message.service.common.InnerEventService;
import com.hp.message.service.common.SendMsgCallBackService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

/**
 * @author 尚肖磊
 *  2021-06-25 10:06
 *  自定义mqtt消息接收类
 */
@Slf4j
@Component
public class ReceMessageHandler implements ApplicationListener<EmqxReceMsgEvent> {

    @Lazy
    @Autowired
    private AsyncCallService asyncCallService;
    @Lazy
    @Autowired
    private InnerEventService innerEventService;
    @Lazy
    @Autowired
    private ISdkReceMsgService sdkReceMsgService;
    @Lazy
    @Autowired
    private SendMsgCallBackService sendMsgCallBackService;

    @Override
    public void onApplicationEvent(EmqxReceMsgEvent emqxReceMsgEvent) {
        // 获取消息主题
        String topic = emqxReceMsgEvent.getEmqxTopic();
        // 获取消息对象
        final EmqxDataMsg emqxReqMsg = emqxReceMsgEvent.getEventData();

        try {
            // 获取消息类型
            MqMsgType mqMsgType = emqxReqMsg.getMsgType();
            // 处理消息响应
            if (!ObjectUtils.isEmpty(emqxReqMsg.getRespConfig())
                    && !StringUtils.isEmpty(emqxReqMsg.getRespConfig().getRespTopic())) {
                // 处理响应消息
                asyncCallService.handleCallable(() -> receMsgResp(emqxReqMsg, mqMsgType));
            }
            asyncCallService.handleCallable(() -> {
                try {
                    // 处理消息业务逻辑
                    switch (mqMsgType) {
                        case EMQX_EQUI_LOGIN_NOTIFY:
                            sdkReceMsgService.receEquiLoginMsg(emqxReqMsg.getEquiSno());
                            break;
                        case EMQX_EQUI_LOGOUT_NOTIFY:
                            EquiLogoutType equiLogoutType = emqxReqMsg.getLogoutCode();
                            sdkReceMsgService.receEquiLogoutMsg(emqxReqMsg.getEquiSno(), equiLogoutType);
                            break;
                        case EMQX_SDK_UPLOAD_DATA_REQ:
                            sdkReceMsgService.receEquiUploadDataMsg(emqxReqMsg.getEquiSno(), emqxReqMsg.getTransferData());
                            break;
                        case EMQX_SDK_PUSH_DATA_RESP:
                            handleSendMsgResp(emqxReqMsg);
                            break;
                        default:
                            break;
                    }
                } catch (Exception ex) {
                    log.error("handler msgId {} exception", emqxReqMsg.getMsgId(), ex);
                }
            });
        } catch (Exception ex) {
            log.error("handler msgId {} exception", emqxReqMsg.getMsgId(), ex);
        }
    }

    /**
     * 统一处理 接收消息响应
     *
     * @param emqxReqMsg 接收消息对象
     * @param mqMsgType    接收消息类型
     */
    private void receMsgResp(EmqxDataMsg emqxReqMsg, MqMsgType mqMsgType) {
        String respEmqxTopic = emqxReqMsg.getRespConfig().getRespTopic();
        EmqxDataMsg emqxDataMsg = new EmqxDataMsg();
        if (!ObjectUtils.isEmpty(emqxReqMsg.getHandleId())) {
            emqxDataMsg.setHandleId(emqxReqMsg.getHandleId());
        }
        emqxDataMsg.setMsgId(emqxReqMsg.getMsgId());
        emqxDataMsg.setEquiSno(emqxReqMsg.getEquiSno());
        switch (mqMsgType) {
            default:
                emqxDataMsg.setMsgType(MqMsgType.EMQX_SDK_UPLOAD_DATA_RESP);
                break;
        }
        emqxDataMsg.setProgressCode(MqMsgProgressedType.HANLDER_FINISH);
        emqxDataMsg.setHandlerResultCode(MqMsgResultType.SUCCESS);
        // 发送响应消息
        EmqxSendMsgEvent emqxSendMsgEvent = new EmqxSendMsgEvent(this, emqxDataMsg, respEmqxTopic);
        innerEventService.publishEvent(emqxSendMsgEvent);
    }

    /**
     * 统一处理 发发送消息响应回调
     *
     * @param emqxReqMsg 消息对象
     */
    private void handleSendMsgResp(EmqxDataMsg emqxReqMsg) throws Exception {
        MsgCallBackWrapper msgCallBackWrapper = sendMsgCallBackService.getMsgCallbackWrapper(emqxReqMsg);
        if (msgCallBackWrapper != null) {
            sendMsgCallBackService.removeMsgCallBack(emqxReqMsg);
            if (emqxReqMsg.getHandlerResultCode().equals(MqMsgResultType.SUCCESS.getResultCode())) {
                msgCallBackWrapper.getMsgCallback().onSuccess(emqxReqMsg, msgCallBackWrapper.getEmqxDataMsg());
            } else {
                MqMsgResultType resultType = emqxReqMsg.getHandlerResultCode();
                msgCallBackWrapper.getMsgCallback().onError(resultType, msgCallBackWrapper.getEmqxDataMsg());
            }
        }
    }

}
