package com.hp.message.domain;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.hp.message.Constant;
import com.hp.message.enums.EmqxQosType;
import com.hp.message.enums.MqMsgType;
import com.hp.message.event.BaseEvent;
import com.hp.message.event.EmqxReceMsgEvent;
import com.hp.message.service.common.AsyncCallService;
import com.hp.message.service.common.InnerEventService;
import com.hp.message.utils.MsgTopicUtil;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.paho.client.mqttv3.*;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;
import org.springframework.util.ObjectUtils;

import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author 尚肖磊
 *  2021-08-23 15:14
 *  emqx 通信客户端类
 */
@Slf4j
public class EmqxClient {

    private EmqxConfig emqxConfig;
    private AsyncCallService asyncCallService;
    private InnerEventService innerEventService;

    private MqttClient mqttClient;
    private MqttConnectOptions mqttConnectOptions;
    private MemoryPersistence memoryPersistence;

    /**
     * 构造方法
     *
     */
    public EmqxClient(EmqxConfig emqxConfig, AsyncCallService asyncCallService, InnerEventService innerEventService) {
        super();
        this.emqxConfig = emqxConfig;
        this.asyncCallService = asyncCallService;
        this.innerEventService = innerEventService;
        this.memoryPersistence = new MemoryPersistence();

        // 初始化连接对象
        initConnectOptions();
        // 启动并连接服务器
        startConnect();
    }

    /**
     * 获取mqtt clientId
     *
     * @return 返回clientId
     */
    private String getClinetId() {
        String clientId = Constant.EMQX_CLIENTID_PREFIX;
        clientId = clientId + System.currentTimeMillis();
        return clientId;
    }

    /**
     * 初始化 mqtt连接参数
     *
     * @return mqtt连接参数对象
     */
    public MqttConnectOptions initConnectOptions() {
        // MQTT的连接设置
        mqttConnectOptions = new MqttConnectOptions();
        // 设置是否清空session,这里如果设置为false表示服务器会保留客户端的连接记录，
        // 这里设置为true表示每次连接到服务器都以新的身份连接
        mqttConnectOptions.setCleanSession(true);
        // 设置认证用户名
        mqttConnectOptions.setUserName(emqxConfig.getAppId());
        // 设置认证密码
        mqttConnectOptions.setPassword(emqxConfig.getAppKey().toCharArray());
        // 设置连接超时时间 单位为秒
        mqttConnectOptions.setConnectionTimeout(emqxConfig.getConnectTimeout());
        // 设置会话心跳时间 单位为秒 服务器会每隔1.5*20秒的时间向客户端发送个消息判断客户端是否在线，但这个方法并没有重连的机制
        mqttConnectOptions.setKeepAliveInterval(emqxConfig.getKeepAliveInterval());
        // 设置打开 mqtt自动重连机制
        mqttConnectOptions.setAutomaticReconnect(true);
        // 配置遗嘱消息
        mqttConnectOptions.setWill(getWillMsgTopic(), getWillMsgContent(), 0, false);
        try {
            mqttClient = new MqttClient(emqxConfig.getMsgHost(), getClinetId(), memoryPersistence);
            return mqttConnectOptions;
        } catch (MqttException ex) {
            log.error("mqtt {} initConnectOptions error", emqxConfig.getAppId(), ex);
            return null;
        }
    }

    /**
     * 获取医嘱消息主题
     *
     * @return
     */
    private String getWillMsgTopic() {
        return MsgTopicUtil.getAppPushTopic(emqxConfig.getAppId());
    }

    /**
     * 获取遗嘱消息对象
     *
     * @return
     */
    private byte[] getWillMsgContent() {
        EmqxDataMsg emqxDataMsg = new EmqxDataMsg();
        emqxDataMsg.setMsgType(MqMsgType.EMQX_SDK_OFFLINE);
        return JSON.toJSONBytes(emqxDataMsg);
    }

    /**
     * 获取订阅主题列表
     *
     * @return
     */
    private List<String> getSubConfigList() {
        List<String> subConfigList = new ArrayList<>();
        subConfigList.add(MsgTopicUtil.getAppDefaultSubTopic(emqxConfig.getAppId()));
        return subConfigList;
    }

    /**
     * mqtt 回调接口对象
     */
    private MqttCallbackExtended mqttCallbackExtended = new MqttCallbackExtended() {

        /**
         * mqtt 连接完成
         *
         * @param reconnect If true, the connection was the result of automatic reconnect.
         * @param serverURI The server URI that the connection was made to.
         */
        @Override
        public void connectComplete(boolean reconnect, String serverURI) {
            if (reconnect) {
                log.info("mqtt {} reconnect success", emqxConfig.getAppId());
            } else {
                log.info("mqtt {} connect success", emqxConfig.getAppId());
            }
            // 如果清除session 需要重新订阅消息主题
            subMessage(getSubConfigList());
        }

        /**
         * mqtt 连接丢失
         *
         * @param throwable the reason behind the loss of connection.
         */
        @Override
        public void connectionLost(Throwable throwable) {
            log.error("mqtt {} connect lost", emqxConfig.getAppId(), throwable);
        }

        /**
         * 消息接收完成
         *
         * @param topic         name of the topic on the message was published to
         * @param mqttMessage   the actual message.
         * @throws Exception    if a terminal error has occurred, and the client should be shut down.
         */
        @Override
        public void messageArrived(String topic, MqttMessage mqttMessage) {
            try {
                EmqxDataMsg receDataMsg = JSON.parseObject(mqttMessage.getPayload(), EmqxDataMsg.class);
                log.debug("mqtt {} receive Message success topic {} data:{}", emqxConfig.getAppId(), topic, JSON.toJSONString(receDataMsg));
                // 发布 接收消息完成事件
                publishEvent(new EmqxReceMsgEvent(this, receDataMsg, topic));
            } catch (Exception ex) {
                log.error("mqtt {} messageArrived exception", emqxConfig.getAppId(), ex);
            }
        }

        /**
         * 消息发送完成
         *
         * @param token the delivery token associated with the message.
         */
        @Override
        public void deliveryComplete(IMqttDeliveryToken token) {
            try {
                String topic = token.getTopics()[0];
                MqttMessage mqttMessage = token.getMessage();
                EmqxDataMsg pushDataMsg = JSON.parseObject(mqttMessage.getPayload(), EmqxDataMsg.class);
                log.debug("mqtt {} push Message success topic {} data:{}", emqxConfig.getAppId(), topic, JSON.toJSONString(pushDataMsg));
            } catch (MqttException ex) {
                log.error("mqtt {} deliveryComplete exception", emqxConfig.getAppId(), ex);
            }
        }
    };

    /**
     * 初始化 mqtt连接服务器
     *
     * @return
     */
    public EmqxResult startConnect() {
        try {
            // Sets the callback listener to use for events that happen asynchronously.
            mqttClient.setCallback(mqttCallbackExtended);
            // start connect mqtt service
            mqttClient.connect(mqttConnectOptions);
            return EmqxResult.builder().isSuccess(true).build();
        } catch (MqttException ex) {
            log.error("mqtt {} connect error", emqxConfig.getAppId(), ex);
            return EmqxResult.builder().isSuccess(false).errorMsg(ex.getMessage()).mqttException(ex).build();
        }
    }

    /**
     * 获取 mqtt连接状态
     *
     * @return
     */
    public boolean getConnectStatus() {
        return !ObjectUtils.isEmpty(mqttClient) ? mqttClient.isConnected() : false;
    }

    /**
     * 发送消息事件
     *
     * @param event 事件对象
     */
    private void publishEvent(BaseEvent event) {
        innerEventService.publishEvent(event);
    }

    /**
     * 断开 mqtt连接状态
     *
     * @return
     */
    public boolean disconnect() {
        if (!ObjectUtils.isEmpty(mqttClient) && mqttClient.isConnected()) {
            try {
                mqttClient.disconnect();
                return true;
            } catch (MqttException ex) {
                log.error("mqtt {} disconnect error", emqxConfig.getAppId(), ex);
                return false;
            }
        }
        return false;
    }

    /**
     * 重连 mqtt服务
     *
     * @return
     */
    public boolean reconnect() {
        try {
            if (!ObjectUtils.isEmpty(mqttClient) && !mqttClient.isConnected()) {
                mqttClient.reconnect();
            }
            return true;
        } catch (MqttException ex) {
            log.error("mqtt {} reconnect error", emqxConfig.getAppId(), ex);
            return false;
        }
    }

    /**
     * 关闭对象并释放所有资源
     *
     * @return
     */
    @PreDestroy
    public boolean close() {
        if (!ObjectUtils.isEmpty(mqttClient)) {
            try {
                if (mqttClient.isConnected()) {
                    // 手动发布 医嘱消息
                    pushEmqxLogoutEvent();
                    mqttClient.disconnect();
                }
                mqttClient.close();
            } catch (MqttException ex) {
                log.error("mqtt {} close error", emqxConfig.getAppId(), ex);
            } finally {
                mqttClient = null;
            }
        }
        log.debug("mqtt {} client close", emqxConfig.getAppId());
        return true;
    }

    /**
     * mqtt 订阅主题
     *
     * @param emqxSubConfigList 订阅主题配置列表
     * @return
     */
    public EmqxResult subMessage(List<String> emqxSubConfigList) {
        if (ObjectUtils.isEmpty(emqxSubConfigList)) {
            log.error("mqtt {} subMessage subConfigList is empty", emqxConfig.getAppId());
            return EmqxResult.builder().isSuccess(false).errorMsg("subConfigList is empty").build();
        }
        if (ObjectUtils.isEmpty(mqttClient) || !mqttClient.isConnected()) {
            log.error("mqtt {} subMessage mqttClient connect error", emqxConfig.getAppId());
            return EmqxResult.builder().isSuccess(false).errorMsg("mqttClient connect error").build();
        }
        for (String item : emqxSubConfigList) {
            try {
                mqttClient.subscribe(item, 0);
                log.debug("mqtt {} subMessage success {}", emqxConfig.getAppId(), item);
            } catch (MqttException ex) {
                log.error("mqtt {} submessage exception {} ", emqxConfig.getAppId(), item, ex);
                return EmqxResult.builder().isSuccess(false)
                        .errorMsg("subscribe topic " + item + " exception").mqttException(ex).build();
            }
        }
        log.info("mqtt {} subMessage success and finish", emqxConfig.getAppId());

        // 延时 发送SDK上线事件消息
        pushEmqxLoginEvent();
        return EmqxResult.builder().isSuccess(true).build();
    }

    /**
     * 构造SDK上线消息
     *
     * @return
     */
    private void pushEmqxLoginEvent() {
        asyncCallService.addDelayTask(() -> {
            String topic = MsgTopicUtil.getAppPushTopic(emqxConfig.getAppId());
            EmqxDataMsg emqxDataMsg = new EmqxDataMsg();
            emqxDataMsg.setMsgType(MqMsgType.EMQX_SDK_ONLINE);
            pushMqttMessage(topic, JSON.toJSONBytes(emqxDataMsg, SerializerFeature.DisableCircularReferenceDetect), EmqxQosType.QOS_0);
        }, 1500, TimeUnit.MILLISECONDS);
    }

    /**
     * 构造SDK上线消息
     *
     * @return
     */
    private void pushEmqxLogoutEvent() {
        String topic = MsgTopicUtil.getAppPushTopic(emqxConfig.getAppId());
        EmqxDataMsg emqxDataMsg = new EmqxDataMsg();
        emqxDataMsg.setMsgType(MqMsgType.EMQX_SDK_OFFLINE);
        pushMqttMessage(topic, JSON.toJSONBytes(emqxDataMsg, SerializerFeature.DisableCircularReferenceDetect), EmqxQosType.QOS_0);
    }

    /**
     * mqtt 取消订阅主题
     *
     * @param emqxSubConfigList 订阅主题配置列表
     * @return
     */
    public EmqxResult unSubMessage(List<String> emqxSubConfigList) {
        if (ObjectUtils.isEmpty(emqxSubConfigList)) {
            log.error("mqtt {} unSubMessage subConfigList is empty", emqxConfig.getAppId());
            return EmqxResult.builder().isSuccess(false).errorMsg("subConfigList is empty").build();
        }
        if (ObjectUtils.isEmpty(mqttClient) || !mqttClient.isConnected()) {
            log.error("mqtt {} unSubMessage mqttClient connect error", emqxConfig.getAppId());
            return EmqxResult.builder().isSuccess(false).errorMsg("mqttClient connect error").build();
        }
        for (String item : emqxSubConfigList) {
            try {
                mqttClient.unsubscribe(item);
                log.info("mqtt {} unSubMessage success {}", emqxConfig.getAppId(), item);
            } catch (MqttException ex) {
                log.error("mqtt {} unSubMessage exception {} ", emqxConfig.getAppId(), item, ex);
                return EmqxResult.builder().isSuccess(false)
                        .errorMsg("unsubscribe topic " + item + " exception").mqttException(ex).build();
            }
        }
        log.info("mqtt {} unSubMessage success", emqxConfig.getAppId());
        return EmqxResult.builder().isSuccess(true).build();
    }

    /**
     * mqtt 向指定主题发布消息
     *
     * @param topic   消息主题
     * @param msgData 消息数据
     * @return
     */
    public EmqxResult pushMqttMessage(String topic, byte[] msgData) {
        return pushMqttMessage(topic, msgData, EmqxQosType.QOS_0);
    }

    /**
     * mqtt 向指定主题发布消息
     *
     * @param topic   消息主题
     * @param msgData 消息数据
     * @param qos     消息qos
     * @return
     */
    public EmqxResult pushMqttMessage(String topic, byte[] msgData, EmqxQosType qos) {
        return pushMqttMessage(topic, msgData, qos, false);
    }

    /**
     * mqtt 向指定主题发布消息
     *
     * @param topic    消息主题
     * @param msgData  消息数据
     * @param qos      消息qos
     * @param retained 消息是否驻留
     * @return
     */
    public synchronized EmqxResult pushMqttMessage(String topic, byte[] msgData, EmqxQosType qos, boolean retained) {
        if (ObjectUtils.isEmpty(mqttClient) || !mqttClient.isConnected()) {
            log.error("mqtt {} pushMqttMessage client connect lost", emqxConfig.getAppId());
            return EmqxResult.builder().isSuccess(false).errorMsg("mqttClient connect lost").build();
        }
        try {
            mqttClient.publish(topic, msgData, qos.getQos(), retained);
            return EmqxResult.builder().isSuccess(true).build();
        } catch (MqttException ex) {
            log.error("mqtt {} pushMqttMessage MqttException", emqxConfig.getAppId(), ex);
            return EmqxResult.builder().isSuccess(false).errorMsg("pushMqttMessage fail").build();
        }
    }

    /**
     * mqtt 向指定主题发布消息
     *
     * @param topic   消息主题
     * @param message 消息对象
     * @return
     */
    public synchronized EmqxResult pushMqttMessage(String topic, MqttMessage message) {
        if (ObjectUtils.isEmpty(mqttClient) || !mqttClient.isConnected()) {
            log.error("mqtt {} pushMqttMessage client connect lost", emqxConfig.getAppId());
            return EmqxResult.builder().isSuccess(false).errorMsg("mqttClient connect lost").build();
        }
        try {
            mqttClient.publish(topic, message);
            return EmqxResult.builder().isSuccess(true).build();
        } catch (MqttException ex) {
            log.error("mqtt {} pushMqttMessage MqttException", emqxConfig.getAppId(), ex);
            return EmqxResult.builder().isSuccess(false).errorMsg("pushMqttMessage fail").build();
        }
    }

}
