package com.hp.message.config;

import com.hp.message.domain.EmqxClient;
import com.hp.message.domain.EmqxConfig;
import com.hp.message.interfaces.ISdkSendMsgService;
import com.hp.message.property.AppProperty;
import com.hp.message.property.MsgProperty;
import com.hp.message.interfaces.ISdkReceMsgService;
import com.hp.message.service.SdkReceMsgService;
import com.hp.message.service.SdkSendMsgService;
import com.hp.message.service.common.AsyncCallService;
import com.hp.message.service.common.InnerEventService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;

/**
 * @author 尚肖磊
 *  2021-06-24 18:11
 *  msg相关自动配置类
 */
@Slf4j
@Configuration
@ComponentScan("com.hp.message.handler")
@AutoConfigureAfter(AppAutoConfigure.class)
@EnableConfigurationProperties(value = MsgProperty.class)
public class MsgAutoConfigure {

    /**
     * 华普app相关配置类
     */
    private AppProperty appProperty;
    /**
     * 华普消息服务配置类
     */
    private final MsgProperty msgProperty;

    public MsgAutoConfigure(MsgProperty msgProperty, AppProperty appProperty) {
        if (msgProperty == null) {
            log.error("{} config lost", MsgProperty.DEFAULT_PREFIX);
        }
        this.msgProperty = msgProperty;
        if (appProperty == null) {
            log.error("{} config lost", AppProperty.DEFAULT_PREFIX);
        }
        this.appProperty = appProperty;
    }

    @Autowired
    private AsyncCallService asyncCallService;
    @Autowired
    private InnerEventService innerEventService;

    /**
     * 构造emqxClient实体对象
     *
     * @return 返回
     */
    @Bean
    @ConditionalOnMissingBean
    public EmqxClient getEmqxConfig() {
        EmqxConfig emqxConfig = new EmqxConfig();
        emqxConfig.setAppId(appProperty.getAppId());
        emqxConfig.setAppKey(appProperty.getAppKey());
        emqxConfig.setMsgHost(msgProperty.getMsgHost());
        emqxConfig.setConnectTimeout(msgProperty.getConnectTimeout());
        emqxConfig.setKeepAliveInterval(msgProperty.getKeepAliveInterval());
        return new EmqxClient(emqxConfig, asyncCallService, innerEventService);
    }

    /**
     * 构造默认消息接收服务类
     *
     * @return 返回
     */
    @Bean
    @ConditionalOnMissingBean
    public ISdkReceMsgService createSdkReceMsService() {
        return new SdkReceMsgService();
    }

    /**
     * 构造消息发送服务类
     *
     * @return 返回
     */
    @Bean
    @ConditionalOnMissingBean
    public ISdkSendMsgService createSdkSendMsService() {
        return new SdkSendMsgService();
    }

}
